<?php

declare(strict_types=1);

namespace Drupal\track_usage;

use Drupal\Core\Config\Entity\ConfigEntityStorageInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\Condition;
use Drupal\Core\Database\Query\ConditionInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Entity\TranslatableInterface;
use Drupal\language\ConfigurableLanguageInterface;
use Drupal\track_usage\Entity\TrackConfigInterface;
use Drupal\track_usage\Model\EntityRole;
use Drupal\track_usage\Model\Usage;
use Drupal\track_usage\Trait\BackwardsCompatibilityTrait;
use Drupal\track_usage\Trait\EntityUtilityTrait;
use Drupal\track_usage\Model\Operation;
use Drupal\track_usage\Trait\IdColumnTrait;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Default implementation of the track_usage.recorder service.
 */
class Recorder implements RecorderInterface, EventSubscriberInterface {

  use BackwardsCompatibilityTrait;
  use EntityUtilityTrait;
  use IdColumnTrait;

  /**
   * List of inserted/updated source entities to be processed later.
   *
   * @var array<non-empty-string, non-empty-string>
   */
  protected array $sources = [];

  /**
   * List of inserted/updated traversable entities to be processed later.
   *
   * @var array<non-empty-string, non-empty-string>
   */
  protected array $traversables = [];

  /**
   * Static cache for track usage config storage.
   */
  protected ConfigEntityStorageInterface $configStorage;

  public function __construct(
    protected readonly TrackerInterface $tracker,
    protected readonly Connection $db,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly EntityFieldManagerInterface $entityFieldManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [KernelEvents::RESPONSE => 'registerUsageRecords'];
  }

  /**
   * {@inheritdoc}
   */
  public function record(EntityInterface $entity, TrackConfigInterface $config): void {
    if ($config->isSource($entity)) {
      $key = $config->id() . ':' . $this->getKey($entity);
      $this->sources[$key] = $key;
    }
    if ($config->isTraversable($entity)) {
      $key = $this->getKey($entity);
      $this->traversables[$key] = $key;
    }
  }

  /**
   * Processes track usage once a response was created.
   */
  public function registerUsageRecords(): void {
    if ($this->traversables) {
      $this->processTraversables();
      $this->traversables = [];
    }

    foreach ($this->sources as $key) {
      [$configId, $key] = explode(':', $key, 2);
      [$entityTypeId, $entityId, $revisionId] = $this->destructKey($key);

      $entity = $this->loadEntity($entityTypeId, $entityId, $revisionId);
      if (!$entity instanceof FieldableEntityInterface) {
        continue;
      }

      // Cleanup first all records for this entity and config.
      $condition = $this->getCondition(entity: $entity, revision: TRUE)
        ->condition('config', $configId);
      $this->deleteUsageRecords($condition);

      $config = $this->getConfigStorage()->load($configId);

      // Insert usage records.
      foreach ($this->getTranslations($entity) as $translation) {
        foreach ($this->tracker->track($translation, $config) as $usage) {
          $this->createUsageRecord($config, $translation, $usage);
        }
      }
    }
    $this->sources = [];
  }

  /**
   * {@inheritdoc}
   */
  public function cleanup(EntityInterface $entity, Operation $operation): void {
    match($operation) {
      Operation::Delete => $this->onEntityDelete($entity),
      Operation::DeleteRevision => $this->onEntityRevisionDelete($entity),
      Operation::DeleteTranslation => $this->onEntityTranslationDelete($entity),
    };

    if (!$entity instanceof FieldableEntityInterface) {
      return;
    }

    // The deleted entity might be a traversable entity, part of usage paths.
    foreach ($this->getConfigStorage()->loadMultiple() as $config) {
      if ($config->isTraversable($entity)) {
        // Queue this traversable entity, it might impact source entities.
        $key = $this->getKey($entity);
        $this->traversables[$key] = $key;
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getTargetsForEntity(FieldableEntityInterface $entity, string $targetEntityTypeId, TrackConfigInterface $config): iterable {
    if (!$config->isSource($entity) && !$config->isTraversable($entity)) {
      return;
    }

    // Find all usage records where this entity appears in the path.
    $query = $this->db->select(self::TABLE_PATHS)
      ->distinct()
      ->fields(self::TABLE, ['target_id_string'])
      ->condition('target_type', $targetEntityTypeId)
      ->condition('type', $entity->getEntityTypeId())
      ->condition('id', $entity->id());
    $query->innerJoin(table: self::TABLE, condition: self::TABLE_PATHS . '.tid = ' . self::TABLE . '.tid');

    foreach ($query->execute()->fetchCol() as $id) {
      yield $id;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getPathsToTarget(EntityInterface $target, TrackConfigInterface $config): array {
    $query = $this->db->select(self::TABLE_PATHS)
      ->fields(self::TABLE_PATHS)
      ->condition('config', $config->id())
      ->condition('target_type', $target->getEntityTypeId())
      ->condition($this->getIdColumnName($target, EntityRole::Target), $this->getIdColumnValueByEntity($target));
    $query->innerJoin(table: self::TABLE, condition: self::TABLE_PATHS . '.tid = ' . self::TABLE . '.tid');

    $paths = [];
    foreach ($query->execute() as $record) {
      $paths["$record->tid:$record->path"][$record->delta] = [$record->type, $record->id, $record->revision];
    }

    return array_values($paths);
  }

  /**
   * Reacts after an entity has been deleted.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity being deleted.
   */
  protected function onEntityDelete(EntityInterface $entity): void {
    // A track usage config was deleted. Remove all its records.
    if ($entity instanceof TrackConfigInterface) {
      $this->deleteUsageRecords((new Condition('AND'))->condition('config', $entity->id()));
      return;
    }
    // A language has been deleted.
    elseif ($entity instanceof ConfigurableLanguageInterface) {
      $this->deleteUsageRecords((new Condition('AND'))->condition('source_langcode', $entity->id()));
      return;
    }

    $entityTypeId = $entity->getEntityTypeId();
    $entityId = $this->getIdColumnValueByEntity($entity);

    $condition = new Condition('OR');
    // The entity acts like a target entity.
    $condition->condition(
      (new Condition('AND'))
        ->condition('target_type', $entityTypeId)
        ->condition($this->getIdColumnName($entityTypeId, EntityRole::Target), $entityId)
    );
    // Or when the entity acts like a source entity.
    $condition->condition(
      (new Condition('AND'))
        ->condition('source_type', $entityTypeId)
        ->condition($this->getIdColumnName($entityTypeId, EntityRole::Source), $entityId)
    );

    $this->deleteUsageRecords($condition);
  }

  /**
   * Reacts after an entity revision has been deleted.
   *
   * @param \Drupal\Core\Entity\EntityInterface $revision
   *   The entity revision being deleted.
   */
  protected function onEntityRevisionDelete(EntityInterface $revision): void {
    $this->deleteUsageRecords($this->getCondition(entity: $revision, revision: TRUE));
  }

  /**
   * Reacts after an entity translation has been deleted.
   *
   * @param \Drupal\Core\Entity\EntityInterface $translation
   *   The entity translation being deleted.
   */
  protected function onEntityTranslationDelete(EntityInterface $translation): void {
    $this->deleteUsageRecords($this->getCondition($translation, TRUE, TRUE));
  }

  /**
   * Returns entity translations.
   *
   * @param \Drupal\Core\Entity\FieldableEntityInterface $entity
   *   The entity.
   *
   * @return iterable<FieldableEntityInterface>
   *   Translated entities.
   */
  protected function getTranslations(FieldableEntityInterface $entity): iterable {
    if ($entity instanceof TranslatableInterface) {
      foreach ($entity->getTranslationLanguages() as $langcode => $language) {
        if ($entity->hasTranslation($langcode)) {
          yield $entity->getTranslation($langcode);
        }
      }
    }
    else {
      yield $entity;
    }
  }

  /**
   * Re-queues source entities if a traversable entity has changed.
   *
   * When a traversable entity is updated or deleted, it might impact the usage
   * of source entities. We search if this traversable entity is part of the
   * usage path and re-queue the source entities where this traversable entity
   * is involved.
   */
  protected function processTraversables(): void {
    // Create an 'OR' condition that would look like:
    // @code
    // (type = 'media' AND id = 123 AND revision = 123)
    //   OR
    // (type = 'media' AND id = 123 AND revision = 10099)
    //   OR
    // (type = 'paragraph' AND id = 999 AND revision = 10099)
    // @endcode
    $or = $this->db->condition('OR');
    foreach ($this->traversables as $key) {
      [$entityTypeId, $entityId, $revisionId] = $this->destructKey($key);
      $or->condition(
        $this->db->condition('AND')
          ->condition('type', $entityTypeId)
          ->condition('id', $entityId)
          ->condition('revision', $revisionId)
      );
    }

    // Get a list of sources where these traversables are involved.
    $query = $this->db->select(self::TABLE_PATHS)
      ->distinct()
      ->fields(self::TABLE, ['config', 'source_type', 'source_id_string', 'source_revision'])
      ->condition($or);
    $query->innerJoin(table: self::TABLE, condition: self::TABLE_PATHS . '.tid = ' . self::TABLE . '.tid');
    $sources = $query->execute()->fetchAll($this->fetchMode(\PDO::FETCH_NUM));

    foreach ($sources as [$configId, $entityTypeId, $entityIdAsString, $revisionId]) {
      $key = "$configId:$entityTypeId:$entityIdAsString:$revisionId";

      // Schedule this source entity for processing if needed.
      if (!isset($this->sources[$key])) {
        $entity = $this->loadEntity($entityTypeId, $entityIdAsString, $revisionId);
        if ($entity) {
          $this->sources[$key] = $key;
        }
      }
    }
  }

  /**
   * Inserts entity usage record in the backend.
   *
   * @param \Drupal\track_usage\Entity\TrackConfigInterface $config
   *   The track usage config.
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity for which to create a new record.
   * @param \Drupal\track_usage\Model\Usage $usage
   *   The usage object.
   */
  protected function createUsageRecord(TrackConfigInterface $config, EntityInterface $entity, Usage $usage): void {
    $tid = $this->db->insert(self::TABLE)->fields([
      'config' => $config->id(),
      'source_type' => $entity->getEntityTypeId(),
      'source_id_int' => $this->getIntegerIdColumnValue($entity),
      'source_id_string' => $this->getStringIdColumnValue($entity),
      'source_langcode' => $entity->language()->getId(),
      'source_revision' => $this->getRevisionId($entity),
      'target_type' => $usage->targetEntity->getEntityTypeId(),
      'target_id_int' => $this->getIntegerIdColumnValue($usage->targetEntity),
      'target_id_string' => $this->getStringIdColumnValue($usage->targetEntity),
    ])->execute();

    if (!$usage->paths) {
      return;
    }

    // Insert path records.
    $insert = $this->db->insert(self::TABLE_PATHS)
      ->fields(['tid', 'path', 'delta', 'type', 'id', 'revision']);

    foreach ($usage->paths as $path => $keys) {
      foreach ($keys as $delta => $key) {
        [$type, $id, $revision] = $this->destructKey($key);
        $insert->values([
          'tid' => $tid,
          'path' => $path,
          'delta' => $delta,
          'type' => $type,
          'id' => $id,
          'revision' => $revision,
        ]);
      }
    }

    $insert->execute();
  }

  /**
   * Deletes usages based on a condition.
   *
   * @param \Drupal\Core\Database\Query\ConditionInterface $condition
   *   A condition to apply against {track_usage} table.
   */
  protected function deleteUsageRecords(ConditionInterface $condition): void {
    $tids = $this->db->select(self::TABLE)
      ->fields(self::TABLE, ['tid'])
      ->condition($condition)
      ->execute()
      ->fetchCol();

    if ($tids) {
      $this->db->delete(self::TABLE_PATHS)->condition('tid', $tids, 'IN')->execute();
    }
    $this->db->delete(self::TABLE)->condition($condition)->execute();
  }

  /**
   * Returns a database query condition to be used when deleting usage records.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   * @param bool $revision
   *   (optional) Condition includes also the revision. Defaults to FALSE.
   * @param bool $langcode
   *   (optional) Condition includes also the language. Defaults to FALSE.
   *
   * @return \Drupal\Core\Database\Query\ConditionInterface
   *   A condition instance.
   */
  protected function getCondition(EntityInterface $entity, bool $revision = FALSE, bool $langcode = FALSE): ConditionInterface {
    $entityTypeId = $entity->getEntityTypeId();
    $entityId = $this->getIdColumnValueByEntity($entity);

    $condition = (new Condition('AND'))
      ->condition('source_type', $entityTypeId)
      ->condition($this->getIdColumnName($entityTypeId, EntityRole::Source), $entityId);

    if ($revision) {
      $condition->condition('source_revision', $this->getRevisionId($entity));
    }

    if ($langcode) {
      $condition->condition('source_langcode', $entity->language()->getId());
    }

    return $condition;
  }

  /**
   * Returns the track usage config entity storage.
   *
   * @return \Drupal\Core\Config\Entity\ConfigEntityStorageInterface
   *   The track usage config entity storage.
   */
  protected function getConfigStorage(): ConfigEntityStorageInterface {
    if (!isset($this->configStorage)) {
      $this->configStorage = $this->entityTypeManager->getStorage('track_usage_config');
    }
    return $this->configStorage;
  }

}
