<?php

declare(strict_types=1);

namespace Drupal\track_usage\Trait;

use Drupal\Component\Utility\DeprecationHelper;
use Drupal\Core\Database\Statement\FetchAs;

/**
 * Backwards-compatibility layer.
 *
 * @todo Remove once Drupal <11.2.0 support is dropped.
 */
trait BackwardsCompatibilityTrait {

  /**
   * Provides a backwards-compatibility layer for PDO fetch mode.
   *
   * @param int $fetchMode
   *   The fetch mode.
   *
   * @return \Drupal\Core\Database\Statement\FetchAs|int
   *   The fetch mode either as an enum or as an integer.
   *
   * @see https://www.drupal.org/node/3488338
   *
   * @phpstan-ignore-next-line
   */
  protected function fetchMode(int $fetchMode): FetchAs|int {
    return DeprecationHelper::backwardsCompatibleCall(
      currentVersion: \Drupal::VERSION,
      deprecatedVersion: '11.2.0',
      currentCallable: function () use ($fetchMode): mixed {
        return match ($fetchMode) {
          // @phpstan-ignore class.notFound
          \PDO::FETCH_ASSOC => FetchAs::Associative,
          // @phpstan-ignore class.notFound
          \PDO::FETCH_CLASS, \PDO::FETCH_CLASS | \PDO::FETCH_PROPS_LATE => FetchAs::ClassObject,
          // @phpstan-ignore class.notFound
          \PDO::FETCH_COLUMN => FetchAs::Column,
          // @phpstan-ignore class.notFound
          \PDO::FETCH_NUM => FetchAs::List,
          // @phpstan-ignore class.notFound
          \PDO::FETCH_OBJ => FetchAs::Object,
          default => throw new \RuntimeException("Fetch mode '$fetchMode' is not supported. Use supported modes only."),
        };
      },
      deprecatedCallable: fn(): int => $fetchMode,
    );
  }

}
