<?php

declare(strict_types=1);

namespace Drupal\Tests\track_usage\Kernel;

use Drupal\Core\Url;
use Drupal\file\Entity\File;
use Drupal\KernelTests\KernelTestBase;
use Drupal\language\Entity\ConfigurableLanguage;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\redirect\Entity\Redirect;
use Drupal\Tests\TestFileCreationTrait;
use Drupal\TestTools\Random;
use Drupal\track_usage\EntityGuesser;

/**
 * @coversDefaultClass \Drupal\track_usage\EntityGuesser
 * @group track_usage
 */
class EntityGuesserTest extends KernelTestBase {

  use TestFileCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'file',
    'language',
    'link',
    'node',
    'path',
    'path_alias',
    'redirect',
    'system',
    'track_usage',
    'track_usage_test',
    'user',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->setSetting('file_public_path', 'sites/default/files');

    $this->installEntitySchema('file');
    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installEntitySchema('path_alias');
    $this->installEntitySchema('redirect');

    $this->installConfig(['language']);
    ConfigurableLanguage::createFromLangcode('fr')->save();
    \Drupal::service('kernel')->rebuildContainer();
    // For some reason, after container rebuilding, the wrappers are gone.
    $this->container->get('stream_wrapper_manager')->register();

    NodeType::create(['type' => 'page'])->save();
  }

  /**
   * @covers ::guessFromUrl
   */
  public function testGuessFromUrl(): void {
    $files = $this->getTestFiles('text');
    File::create(['fid' => 1, 'uri' => $files[0]->uri])->save();
    File::create(['fid' => 2, 'uri' => $files[1]->uri])->save();

    Node::create([
      'nid' => 1,
      'type' => 'page',
      'title' => Random::machineName(),
    ])->save();
    Node::create([
      'nid' => 2,
      'type' => 'page',
      'title' => $this->randomMachineName(),
      'path' => ['alias' => '/test/node/number/two'],
    ])->save();
    Node::create([
      'nid' => 3,
      'type' => 'page',
      'title' => $this->randomMachineName(),
    ])->save();
    $redirect = Redirect::create();
    $redirect->setSource('this/is/a/redirect');
    $redirect->setRedirect(Node::load(3)->toUrl()->toString());
    $redirect->setLanguage('fr');
    $redirect->save();

    foreach ($this->getTestCases() as $delta => $case) {
      [$url, $expectancy] = $case;
      $entity = $this->container->get(EntityGuesser::class)->guessFromUrl($url);
      $message = "Row $delta, data: " . var_export($case, TRUE);
      if ($expectancy === NULL) {
        $this->assertNull($entity, $message);
      }
      else {
        $this->assertEquals($expectancy, [$entity->getEntityTypeId(), $entity->id()], $message);
      }
    }
  }

  /**
   * Provides test cases for self::testGuessFromUrl().
   *
   * @return iterable<array{\Drupal\Core\Url|string, array{non-empty-string, positive-int}}>
   *   Test cases.
   */
  public function getTestCases(): iterable {
    $generator = $this->container->get('file_url_generator');
    $fileUri = File::load(2)->getFileUri();

    // Malformed external URL.
    yield ['http://example dot com', NULL];

    // External URL.
    yield ['http://example.com', NULL];

    // Invalid URL (mail).
    yield ['mailto:example@example.com', NULL];
    yield ['mailto:example@example.com?subject=The quick brown fox jumps over the lazy dog', NULL];

    // Non-existing URL.
    yield ['example@example.com', NULL];

    // Routed URL.
    yield [Node::load(1)->toUrl()->toString(), ['node', 1]];

    // Node URL. No alias.
    yield ['/node/1', ['node', 1]];
    yield [Url::fromUserInput('/node/1')->toString(), ['node', 1]];
    yield ['internal:/node/1', ['node', 1]];
    yield [Url::fromUri('internal:/node/1')->toString(), ['node', 1]];
    yield ['node/1', ['node', 1]];

    // Node URL. No alias. Absolute.
    yield [$GLOBALS['base_url'] . '/node/1', ['node', 1]];

    // Drupal URI with internal: scheme.
    yield ['internal:/' . Node::load(1)->toUrl()->getInternalPath(), ['node', 1]];

    // Drupal URI with entity: scheme.
    yield ['entity:node/2', ['node', 2]];

    // Drupal URI with base: scheme.
    yield ['base:' . $generator->generateString($fileUri), ['file', 2]];

    // Drupal URI with route: scheme.
    yield ['route:entity.node.canonical;node=3', ['node', 3]];

    // Aliased node URL.
    yield ['/test/node/number/two', ['node', 2]];
    yield ['internal:/test/node/number/two', ['node', 2]];

    // Aliased absolute node URL.
    yield [$GLOBALS['base_url'] . '/test/node/number/two', ['node', 2]];

    // Redirect with language prefix translated.
    yield ['/fr/this/is/a/redirect', ['node', 3]];
    yield ['internal:/fr/this/is/a/redirect', ['node', 3]];

    // File URI.
    yield [File::load(1)->getFileUri(), ['file', 1]];

    // A path to a file.
    yield [$generator->generateString($fileUri), ['file', 2]];
    yield ['base:' . $generator->generateString($fileUri), ['file', 2]];

    // An absolute file URL.
    yield [$generator->generateAbsoluteString($fileUri), ['file', 2]];

    // Test 3rd-party hook implementation.
    yield ['/abc', ['node', 1]];
    yield ['internal:/abc', ['node', 1]];
    yield ['base:abc', ['node', 1]];
    yield [$GLOBALS['base_url'] . '/abc', ['node', 1]];
  }

}
