<?php

declare(strict_types=1);

namespace Drupal\Tests\track_usage\Kernel\Track;

use Drupal\entity_test\Entity\EntityTest;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\filter\Entity\FilterFormat;
use Drupal\media\Entity\Media;
use Drupal\media\MediaInterface;
use Drupal\Tests\media\Traits\MediaTypeCreationTrait;
use Drupal\Tests\TestFileCreationTrait;

/**
 * Tests the 'media_embed' track plugin.
 *
 * @group track_usage
 */
class MediaEmbedTest extends TrackTestBase {

  use MediaTypeCreationTrait;
  use TestFileCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'filter',
    'image',
    'media',
    'text',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('entity_test');
    $this->installEntitySchema('media');
    $this->installEntitySchema('file');
    $this->installSchema('file', ['file_usage']);

    $this->createMediaType('image', ['id' => 'test']);

    FieldStorageConfig::create([
      'field_name' => 'text',
      'type' => 'text_with_summary',
      'entity_type' => 'entity_test',
      'cardinality' => 1,
    ])->save();
    FieldConfig::create([
      'field_name' => 'text',
      'label' => $this->randomString(),
      'entity_type' => 'entity_test',
      'bundle' => 'entity_test',
    ])->save();
    FilterFormat::create([
      'format' => 'basic_html',
      'name' => 'Basic HTML',
    ])->setFilterConfig('filter_html', [
      'status' => TRUE,
      'settings' => [
        'allowed_html' => '<drupal-media data-entity-uuid data-entity-type>',
      ],
    ])->save();
  }

  /**
   * @covers \Drupal\track_usage\Plugin\TrackUsage\Track\MediaEmbed
   */
  public function testPlugin(): void {
    $media1 = Media::create([
      'bundle' => 'test',
      'name' => $this->randomString(),
    ]);
    $media1->save();
    $media2 = Media::create([
      'bundle' => 'test',
      'name' => $this->randomString(),
    ]);
    $media2->save();

    $text = $this->randomString() .
      $this->buildMediaEmbedTag($media1) . $this->randomString();
    $summary = $this->randomString() .
      $this->buildMediaEmbedTag($media2) . $this->randomString();

    $entity = EntityTest::create([
      'type' => 'entity_test',
      'name' => $this->randomString(),
      'text' => [
        'value' => $text,
        'summary' => $summary,
        'format' => 'basic_html',
      ],
    ]);
    $entity->save();

    $this->assertTargetEntities('media_embed', $entity, 'text', [
      $media1,
      $media2,
    ]);
  }

  /**
   * Builds a media embed HTML tag.
   *
   * @param \Drupal\media\MediaInterface $media
   *   The media entity.
   *
   * @return string
   *   HTML tag.
   */
  protected function buildMediaEmbedTag(MediaInterface $media): string {
    return '<drupal-media data-entity-type="media" data-entity-uuid="' . $media->uuid() . '"></drupal-media>';
  }

}
