<?php

declare(strict_types=1);

namespace Drupal\Tests\track_usage\Traits;

use Drupal\Core\File\FileSystemInterface;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\field\FieldStorageConfigInterface;
use Drupal\file\Entity\File;
use Drupal\language\Entity\ConfigurableLanguage;
use Drupal\media\Entity\Media;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\paragraphs\Entity\Paragraph;
use Drupal\Tests\field\Traits\EntityReferenceFieldCreationTrait;
use Drupal\Tests\media\Traits\MediaTypeCreationTrait;
use Drupal\Tests\paragraphs\FunctionalJavascript\ParagraphsTestBaseTrait;

/**
 * Creates testing data structure and content.
 */
trait TestingDataTrait {

  use EntityReferenceFieldCreationTrait;
  use MediaTypeCreationTrait;
  use ParagraphsTestBaseTrait;

  /**
   * Testing files.
   *
   * @var array<int, \Drupal\file\FileInterface>
   */
  protected array $file = [];

  /**
   * Testing entities.
   *
   * @var array<string, array<int, \Drupal\Core\Entity\ContentEntityInterface>>
   */
  protected array $entity = [];

  /**
   * Create node structure.
   */
  protected function createDataStructure(): void {
    ConfigurableLanguage::createFromLangcode('ro')->save();

    $nodeType = NodeType::create(['type' => 'page']);
    $nodeType->save();

    if (!FieldStorageConfig::loadByName('node', 'body')) {
      FieldStorageConfig::create([
        'field_name' => 'body',
        'type' => 'text_with_summary',
        'entity_type' => 'node',
      ])->save();
    }
    FieldConfig::create([
      'entity_type' => 'node',
      'bundle' => 'page',
      'field_name' => 'body',
      'label' => 'Body',
    ])->save();

    FieldStorageConfig::create([
      'type' => 'file',
      'entity_type' => 'node',
      'field_name' => 'file',
    ])->setCardinality(FieldStorageConfigInterface::CARDINALITY_UNLIMITED)
      ->save();
    FieldConfig::create([
      'entity_type' => 'node',
      'bundle' => 'page',
      'field_name' => 'file',
      'label' => 'File',
      'settings' => [
        'file_extensions' => 'txt html',
      ],
    ])->save();

    FieldStorageConfig::create([
      'entity_type' => 'node',
      'field_name' => 'image',
      'type' => 'image',
    ])->save();
    FieldConfig::create([
      'entity_type' => 'node',
      'bundle' => 'page',
      'field_name' => 'image',
    ])->save();

    // Paragraph structure.
    $this->addParagraphsType('top_paragraph');
    $this->addParagraphsType('nested_paragraph');
    $this->addParagraphsField('top_paragraph', 'nested_paragraph_field', 'paragraph');
    $this->addFieldtoParagraphType('nested_paragraph', 'text', 'text_with_summary');
    FieldStorageConfig::create([
      'field_name' => 'top_paragraph_field',
      'entity_type' => 'node',
      'type' => 'entity_reference_revisions',
      'cardinality' => '-1',
      'settings' => ['target_type' => 'paragraph'],
    ])->save();
    FieldConfig::create([
      'entity_type' => 'node',
      'bundle' => 'page',
      'field_name' => 'top_paragraph_field',
      'settings' => [
        'handler' => 'default:paragraph',
        'handler_settings' => ['target_bundles' => NULL],
      ],
    ])->save();

    // Media setup.
    $this->createMediaType('file', ['id' => 'file']);
    $this->addFieldtoParagraphType('nested_paragraph', 'paragraph_media_file', 'entity_reference', [
      'target_type' => 'media',
    ]);
    $this->createEntityReferenceField('node', 'page', 'media_file', 'Media file', 'media');
  }

  /**
   * Creates testing content.
   *
   * Will create a node with this structure:
   * @code
   * content: (node):
   *   type: page
   *   file: 1 (file)
   *   image: 2 (image)
   *   body:
   *     - 3 (file, embedded)
   *     - 4 (image, embedded)
   *   top_paragraph_field:
   *     top_paragraph (paragraph):
   *       id: 2
   *       nested_paragraph_field:
   *         nested_paragraph (paragraph):
   *           id: 1
   *           text:
   *             - 4 (image, embedded)
   *             - 5 (file, embedded)
   *           paragraph_media_file:
   *             media: (media)
   *               mid: 1
   *               field_media_file: 6 (file)
   *   media_file:
   *     mid: 2
   *     field_media_file:
   *       field_media_file: 5 (file)
   * @endcode
   */
  protected function createData(): void {
    $this->entity['node'][1] = Node::create([
      'nid' => 1,
      'vid' => 1,
      'type' => 'page',
      'title' => $this->randomString(),
      'file' => $this->file[1],
      'image' => $this->file[2],
      'body' => [
        'value' => $this->buildBody([3], [4]),
        'format' => 'basic_html',
      ],
      'top_paragraph_field' => Paragraph::create([
        'id' => 2,
        'revision_id' => 2,
        'type' => 'top_paragraph',
        'nested_paragraph_field' => Paragraph::create([
          'id' => 1,
          'revision_id' => 1,
          'type' => 'nested_paragraph',
          'text' => $this->buildBody([5], [4]),
          'paragraph_media_file' => Media::create([
            'mid' => 1,
            'bundle' => 'file',
            'field_media_file' => $this->file[6],
          ]),
        ]),
      ]),
      'media_file' => Media::create([
        'mid' => 2,
        'vid' => 2,
        'bundle' => 'file',
        'field_media_file' => $this->file[5],
      ]),
      'status' => FALSE,
    ]);

    $this->entity['node'][1]->addTranslation('ro', ['title' => $this->randomString()] + $this->entity['node'][1]->toArray());
    $this->entity['node'][1]->getTranslation('ro')->get('top_paragraph_field')->appendItem(
      Paragraph::create([
        'type' => 'top_paragraph',
        'id' => 3,
        'revision_id' => 3,
        'nested_paragraph_field' => Paragraph::create([
          'id' => 4,
          'revision_id' => 4,
          'type' => 'nested_paragraph',
          'text' => $this->buildBody([1], [6]),
        ]),
      ]),
    );
    $this->entity['node'][1]->save();
  }

  /**
   * Create file entities.
   */
  protected function createFiles(): void {
    /** @var \Drupal\Core\File\FileSystemInterface $fileSystem */
    $fileSystem = \Drupal::service('file_system');
    $destinationDir = 'public://path/to';
    $sourceDir = DRUPAL_ROOT . '/core/tests/fixtures/files';
    $fileSystem->prepareDirectory($destinationDir, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);
    $this->assertDirectoryExists($destinationDir);
    foreach ($this->getTestingFiles() as $fid => $file) {
      $fileUri = "$destinationDir/$file";
      $fileSystem->copy("$sourceDir/$file", $fileUri);
      $this->assertFileExists($fileUri);
      $this->file[$fid] = File::create([
        'fid' => $fid,
        'uri' => $fileUri,
        'status' => TRUE,
      ]);
      $this->file[$fid]->save();
    }
  }

  /**
   * Returns a list of testing files.
   *
   * @return string[]
   *   A list of testing files
   */
  protected function getTestingFiles(): array {
    return [
      1 => 'html-1.txt',
      2 => 'image-1.png',
      3 => 'README.txt',
      4 => 'image-test.png',
      5 => 'html-2.html',
      6 => 'sql-1.txt',
    ];
  }

  /**
   * Builds a testing body string.
   *
   * @param list<int> $files
   *   The file IDs.
   * @param list<int> $images
   *   The image file IDs.
   *
   * @return string
   *   Node body string.
   */
  protected function buildBody(array $files = [], array $images = []): string {
    $fileUrlGenerator = $this->container->get('file_url_generator');

    $body = [];
    foreach ($files as $fid) {
      $body[] = '<a data-entity-type="file" data-entity-uuid="' . $this->file[$fid]->uuid() . '" href="' . $fileUrlGenerator->generateAbsoluteString($this->file[$fid]->getFileUri()) . '">' . $this->randomString() . '</a> ';
      $body[] = $this->randomString(20);
    }
    foreach ($images as $fid) {
      $body[] = ' <img data-entity-type="' . $this->file[$fid]->getEntityTypeId() . '" data-entity-uuid="' . $this->file[$fid]->uuid() . '" src="' . $fileUrlGenerator->generateAbsoluteString($this->file[$fid]->getFileUri()) . '">';
      $body[] = $this->randomString(20);
    }

    return implode("\n", $body);
  }

}
