<?php

namespace Drupal\transform_api_views\Transform;

use Drupal\transform_api\Transform\PluginTransformBase;

/**
 * A transform of a view.
 */
class ViewTransform extends PluginTransformBase {

  /**
   * Construct a ViewsTransform.
   *
   * @param string $view_id
   *   The view id of the view.
   * @param string $display_id
   *   Display id of the view.
   * @param array $options
   *   (Optional) Options to be used with the view.
   * @param array $args
   *   (Optional) Arguments to be used with the view.
   */
  public function __construct($view_id, $display_id, $options = [], $args = []) {
    $this->values = [
      'view_id' => $view_id,
      'display_id' => $display_id,
      'options' => $options,
      'args' => $args,
    ];
    $this->cacheTags[] = 'view:' . $view_id;
  }

  /**
   * Returns the view id of the view.
   *
   * @return string
   *   The view id of the view.
   */
  public function getViewId(): string {
    return $this->getValue('view_id');
  }

  /**
   * Returns the display id of the view.
   *
   * @return string
   *   The display id of the view.
   */
  public function getDisplayId(): string {
    return $this->getValue('display_id');
  }

  /**
   * Returns the options for the view.
   *
   * @return array
   *   The options for the view.
   */
  public function getOptions(): array {
    return $this->getValue('options');
  }

  /**
   * Returns the value of an option for the view.
   *
   * @param string $key
   *   The key of the option.
   *
   * @return mixed
   *   The value of the option.
   */
  public function getOption(string $key): mixed {
    return $this->getValue('options')[$key] ?? NULL;
  }

  /**
   * Returns the arguments for the view.
   *
   * @return array
   *   The arguments for the view.
   */
  public function getArguments(): array {
    return $this->getValue('args');
  }

  /**
   * {@inheritdoc}
   */
  public function getTransformType() {
    return 'view';
  }

  /**
   * {@inheritdoc}
   */
  public function getAlterIdentifiers() {
    return [
      $this->getTransformType(),
      $this->getTransformType() . '_' . $this->values['view_id'],
      $this->getTransformType() . '_' . $this->values['view_id'] . '_' . $this->values['display_id'],
    ];
  }

}
