<?php

namespace Drupal\transform_api\Event;

use Drupal\Component\EventDispatcher\Event;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\transform_api\Transform\TransformInterface;

/**
 * This event allows you to adapt different types of main content.
 *
 * Call setTransform() to set the transform that will be transformed for the
 * current request. The propagation of this event is stopped as soon as
 * a transform is set.
 */
class TransformMainContentEvent extends Event {

  /**
   * The url for the current request.
   *
   * @var string
   */
  private $url;

  /**
   * Route match for the current request.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  private RouteMatchInterface $routeMatch;

  /**
   * Caching metadata for the current request.
   *
   * @var \Drupal\Core\Cache\CacheableMetadata
   */
  private CacheableMetadata $cacheMetadata;

  /**
   * The transform used for the main content of the current request.
   *
   * @var \Drupal\transform_api\Transform\TransformInterface|null
   */
  private ?TransformInterface $transform = NULL;

  public function __construct($url, RouteMatchInterface $routeMatch, CacheableMetadata $cacheMetadata) {
    $this->url = $url;
    $this->routeMatch = $routeMatch;
    $this->cacheMetadata = $cacheMetadata;
  }

  /**
   * Returns the url of the current request.
   *
   * @return string
   *   The url of the current request.
   */
  public function getUrl() {
    return $this->url;
  }

  /**
   * Returns the router match of the current request.
   *
   * @return \Drupal\Core\Routing\RouteMatchInterface
   *   The router match of the current request.
   */
  public function getRouteMatch(): RouteMatchInterface {
    return $this->routeMatch;
  }

  /**
   * Returns the cacheable metadata for the main content.
   *
   * @return \Drupal\Core\Cache\CacheableMetadata
   *   The cacheable metadata for the main content.
   */
  public function getCacheMetadata(): CacheableMetadata {
    return $this->cacheMetadata;
  }

  /**
   * Set the cacheable metadata for the main content.
   *
   * @param \Drupal\Core\Cache\CacheableMetadata $cacheMetadata
   *   The cacheable metadata for the main content.
   */
  public function setCacheMetadata(CacheableMetadata $cacheMetadata): void {
    $this->cacheMetadata = $cacheMetadata;
  }

  /**
   * Returns the transform for the main content.
   *
   * @return \Drupal\transform_api\Transform\TransformInterface|null
   *   The transform for the main content or NULL.
   */
  public function getTransform(): ?TransformInterface {
    return $this->transform;
  }

  /**
   * Set the transform for the main content.
   *
   * @param \Drupal\transform_api\Transform\TransformInterface $transform
   *   The transform for the main content.
   */
  public function setTransform(TransformInterface $transform): void {
    $this->transform = $transform;

    $this->stopPropagation();
  }

}
