<?php

namespace Drupal\transform_api_views\Transform;

use Drupal\Core\Url;
use Drupal\transform_api\Transform\PagerTransform;
use Drupal\views\ViewExecutable;

/**
 * A transform for a views mini pager.
 */
class ViewsMiniPagerTransform extends PagerTransform {

  /**
   * The view that the pager is attached to.
   *
   * @var \Drupal\views\ViewExecutable
   */
  protected ViewExecutable $view;

  /**
   * Whether the pager links should be absolute.
   *
   * @var bool
   */
  protected bool $absoluteLinks = FALSE;

  /**
   * Link destination.
   *
   * @var string
   */
  protected string $linkDestination = 'current';

  /**
   * Construct a ViewsMiniPagerTransform.
   *
   * @param int $element
   *   The pager element that should be transformed.
   * @param \Drupal\views\ViewExecutable|null $view
   *   The view that the pager is attached to.
   *
   * @throws \Exception
   */
  public function __construct($element = 0, ViewExecutable $view = NULL) {
    parent::__construct($element);

    if ($view == NULL) {
      throw new \Exception('View is required for ViewsMiniPagerTransform');
    }

    $this->view = $view;

    $this->absoluteLinks = $this->hasAbsoluteLinks();
    $this->linkDestination = $this->getLinkDestination();
  }

  /**
   * {@inheritDoc}
   */
  public function transform() {
    $build = parent::transform();

    $build['previous'] = $this->getPreviousPageLink();
    $build['next'] = $this->getNextPageLink();

    return $build;
  }

  /**
   * Return the previous page link.
   *
   * @return string|null
   *   The previous page link.
   *
   * @throws \Exception
   */
  protected function getPreviousPageLink() {
    if ($this->getValue('current') === NULL) {
      return NULL;
    }

    if ($this->getValue('current') > 0) {
      $previousPage = $this->getValue('current') - 1;
      return $this->getLink($previousPage);
    }

    return NULL;
  }

  /**
   * Return the next page link.
   *
   * @return \Drupal\Core\GeneratedUrl|string|null
   *   The next page link.
   *
   * @throws \Exception
   */
  protected function getNextPageLink() {
    if ($this->getValue('current') === NULL || $this->getValue('pages') === NULL) {
      return NULL;
    }

    if ($this->getValue('current') < $this->getValue('pages') - 1) {
      $nextPage = $this->getValue('current') + 1;
      return $this->getLink($nextPage);
    }

    return NULL;
  }

  /**
   * Return the link for a given page.
   *
   * @param string|int $page
   *   The page number.
   *
   * @return \Drupal\Core\GeneratedUrl|string
   *   The link for the given page.
   *
   * @throws \Exception
   */
  protected function getLink($page) {
    $url = match ($this->linkDestination) {
      'current' => Url::fromRoute('<current>', ['page' => $page]),
      'results' => Url::fromRoute('transform_api_views.view_results', [
        'view_id' => $this->view->id(),
        'display_id' => $this->view->current_display,
        'page' => $page,
      ]),
      default => throw new \Exception('Invalid link destination'),
    };

    if ($this->linkDestination == 'current') {
      $current_options['query']['format'] = 'json';

      $region = \Drupal::request()->get('region');

      if ($region !== NULL) {
        $current_options['query']['region'] = $region;
      }

      $url->mergeOptions($current_options);
    }

    $url->mergeOptions([
      'query' => $this->view->getExposedInput(),
    ]);

    $url->setAbsolute($this->absoluteLinks);

    return $url->toString();
  }

  /**
   * Check if the pager links should be absolute.
   *
   * @return bool
   *   Whether the pager links should be absolute.
   */
  protected function hasAbsoluteLinks() {
    $pager_transform_links = $this->view->getDisplay()
      ->getExtenders()['transform_api_views']->options['pager_transform_links'] ?? 'relative';

    if ($pager_transform_links == 'absolute') {
      return TRUE;
    }

    return FALSE;
  }

  /**
   * Get the link destination.
   *
   * @return string
   *   The link destination.
   */
  protected function getLinkDestination() {
    $pager_transform_links_destination = $this->view->getDisplay()
      ->getExtenders()['transform_api_views']->options['pager_transform_links_destination'] ?? 'current';

    return $pager_transform_links_destination;
  }

}
