<?php

namespace Drupal\transform_api\Plugin\Transform\Block;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\transform_api\Event\TransformMainContentEvent;
use Drupal\transform_api\Transform\RouteTransform;
use Drupal\transform_api\TransformBlockBase;
use Drupal\transform_api\TransformEvents;
use Psr\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides a 'Main page content' block.
 *
 * @TransformBlock(
 *   id = "system_main_block",
 *   admin_label = @Translation("Main page content"),
 *   category = @Translation("System"),
 * )
 */
class SystemMainTransformBlock extends TransformBlockBase {

  /**
   * The current route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected RouteMatchInterface $routeMatch;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * The event dispatcher.
   *
   * @var \Psr\EventDispatcher\EventDispatcherInterface
   */
  private EventDispatcherInterface $eventDispatcher;

  /**
   * Constructs a SystemMainTransformBlock object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Routing\RouteMatchInterface $routeMatch
   *   The current route match.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   * @param \Psr\EventDispatcher\EventDispatcherInterface $eventDispatcher
   *   The event dispatcher.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, RouteMatchInterface $routeMatch, RequestStack $requestStack, EventDispatcherInterface $eventDispatcher) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->routeMatch = $routeMatch;
    $this->requestStack = $requestStack;
    $this->eventDispatcher = $eventDispatcher;
  }

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static($configuration, $plugin_id, $plugin_definition,
      $container->get('current_route_match'),
      $container->get('request_stack'),
      $container->get('event_dispatcher')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function transform() {
    $url = $this->requestStack->getCurrentRequest()->getSchemeAndHttpHost() . $this->requestStack->getCurrentRequest()->getBaseUrl() . $this->requestStack->getCurrentRequest()->getPathInfo();
    $cacheMetadata = new CacheableMetadata();
    $cacheMetadata->setCacheContexts(['route']);

    $event = new TransformMainContentEvent($url, $this->routeMatch, $cacheMetadata);
    $this->eventDispatcher->dispatch($event, TransformEvents::MAIN_CONTENT);

    if (!is_null($event->getTransform())) {
      $cacheMetadata = $event->getCacheMetadata();
      $transform = [
        '#collapse' => TRUE,
        'content' => $event->getTransform(),
      ];
    }
    else {
      /** @var \Symfony\Component\HttpFoundation\ParameterBag $parameter_bag */
      $parameter_bag = $this->routeMatch->getRawParameters();
      $transform = [
        '#collapse' => TRUE,
        'content' => new RouteTransform($this->routeMatch->getRouteName(), $parameter_bag->all()),
      ];
    }
    $cacheMetadata->applyTo($transform);
    return $transform;
  }

}
