<?php

namespace Drupal\transform_api\EventSubscriber;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Cache\VariationCacheInterface;
use Drupal\Core\Render\RenderCacheInterface;
use Drupal\transform_api\Transform\TransformInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\TerminateEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Caches transformations after the response has been sent.
 */
class TransformationCache implements EventSubscriberInterface {

  /**
   * The variation cache.
   *
   * @var \Drupal\Core\Cache\VariationCacheInterface
   */
  protected $cache;

  /**
   * The things to cache after the response has been sent.
   *
   * @var array
   */
  protected $toCache = [];

  /**
   * The default cache contexts to vary every cache item by.
   *
   * @var string[]
   */
  protected $cacheContexts = [];

  /**
   * Constructs a new TransformationCache object.
   *
   * @param \Drupal\Core\Cache\VariationCacheInterface $cache
   *   The variation cache.
   */
  public function __construct(VariationCacheInterface $cache) {
    $this->cache = $cache;
  }

  /**
   * Reads an entity normalization from cache.
   *
   * The returned normalization may only be a partial transformation because it
   * was previously transformed with a sparse fieldset.
   *
   * @param \Drupal\transform_api\Transform\TransformInterface $object
   *   The transform object for which to generate a cache item.
   *
   * @return array|false
   *   The cached transformation, or FALSE if not yet cached.
   *
   * @see \Drupal\dynamic_page_cache\EventSubscriber\DynamicPageCacheSubscriber::renderArrayToResponse()
   */
  public function get(TransformInterface $object) {
    $cached = $this->cache->get($object->getCacheKeys(), (new CacheableMetadata())->setCacheContexts($this->cacheContexts));
    return $cached ? $cached->data : FALSE;
  }

  /**
   * Adds a transformation to be cached after the response has been sent.
   *
   * @param \Drupal\transform_api\Transform\TransformInterface $object
   *   The transform object for which to generate a cache item.
   * @param array $transformation
   *   The transformation to cache.
   */
  public function saveOnTerminate(TransformInterface $object, array $transformation) {
    $key = implode(':', $object->getCacheKeys());
    $this->toCache[$key] = [$object, $transformation];
  }

  /**
   * Writes transformations to cache, if any were created.
   *
   * @param \Symfony\Component\HttpKernel\Event\TerminateEvent $event
   *   The Event to process.
   */
  public function onTerminate(TerminateEvent $event) {
    foreach ($this->toCache as $value) {
      [$object, $transformation] = $value;
      $this->set($object, $transformation);
    }
  }

  /**
   * Writes a normalization to cache.
   *
   * @param \Drupal\transform_api\Transform\TransformInterface $object
   *   The resource object for which to generate a cache item.
   * @param array $transformation
   *   The transformation to cache.
   *
   * @see \Drupal\dynamic_page_cache\EventSubscriber\DynamicPageCacheSubscriber::responseToRenderArray()
   * @todo Refactor/remove once https://www.drupal.org/node/2551419 lands.
   */
  protected function set(TransformInterface $object, array $transformation) {
    $this->cache->set($object->getCacheKeys(), $transformation, CacheableMetadata::createFromRenderArray($transformation), (new CacheableMetadata())->setCacheContexts($this->cacheContexts));
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events[KernelEvents::TERMINATE][] = ['onTerminate'];
    return $events;
  }

}
