<?php

namespace Drupal\transform_api\Plugin\Transform\Block;

use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Security\TrustedCallbackInterface;
use Drupal\transform_api\TransformBlockBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a messages block.
 *
 * @TransformBlock(
 *   id = "system_messages_block",
 *   admin_label = @Translation("Messages"),
 *   category = @Translation("System"),
 * )
 */
class SystemMessagesTransformBlock extends TransformBlockBase implements TrustedCallbackInterface{

  /**
   * Constructs a new SystemBreadcrumbBlock object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, MessengerInterface $messenger) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static($configuration, $plugin_id, $plugin_definition, $container->get('messenger'));
  }

  /**
   * {@inheritdoc}
   */
  public function transform() {
    return [
      '#lazy_transformer' => [
        static::class . '::transformMessages', [],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks() {
    return ['transformMessages'];
  }

  /**
   * #lazy_transformer callback; replaces placeholder with messages.
   *
   * @param string|null $type
   *   Limit the messages returned by type. Defaults to NULL, meaning all types.
   *   Passed on to \Drupal\Core\Messenger\Messenger::deleteByType(). These
   *   values are supported:
   *   - NULL
   *   - 'status'
   *   - 'warning'
   *   - 'error'.
   *
   * @return array
   *   A transform array containing the messages.
   *
   * @see \Drupal\Core\Messenger\Messenger::deleteByType()
   */
  public static function transformMessages($type = NULL) {
    $transform = [];
    if (isset($type)) {
      $messages = [
        $type => \Drupal::messenger()->deleteByType($type),
      ];
    }
    else {
      $messages = \Drupal::messenger()->deleteAll();
    }

    if ($messages) {
      // Transform the messages.
      $transform = [
        '#type' => 'status_messages',
        '#message_list' => $messages,
        '#status_headings' => [
          'status' => t('Status message'),
          'error' => t('Error message'),
          'warning' => t('Warning message'),
        ],
      ];
    }
    return $transform;
  }

}
