<?php

namespace Drupal\transform_api_views\Plugin\views\display_extender;

use Drupal\Core\Form\FormStateInterface;
use Drupal\transform_api\Repository\EntityTransformRepositoryInterface;
use Drupal\views\Plugin\views\display_extender\DisplayExtenderPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Display extender plugin to control Transform settings.
 *
 * @ingroup views_display_extender_plugins
 *
 * @ViewsDisplayExtender(
 *   id = "transform_api_views",
 *   title = @Translation("Transform API Views"),
 *   help = @Translation("Manage transform settings."),
 *   no_ui = FALSE,
 * )
 */
class TransformApiViews extends DisplayExtenderPluginBase {

  /**
   * The entity transform repository.
   *
   * @var \Drupal\transform_api\Repository\EntityTransformRepositoryInterface
   */
  protected EntityTransformRepositoryInterface $entityTransformRepository;

  /**
   * Constructs a new TransformApiViews object.
   *
   * @param array $configuration
   *   The configuration.
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param \Drupal\transform_api\Repository\EntityTransformRepositoryInterface $entityTransformRepository
   *   The entity transform repository.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTransformRepositoryInterface $entityTransformRepository) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->entityTransformRepository = $entityTransformRepository;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('transform_api.entity_display.repository')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defineOptions() {
    return [
      'row_transform' => [
        'default' => 'entity',
      ],
      'row_transform_mode' => [
        'default' => 'default',
      ],
      'pager_transform' => [
        'default' => 'basic',
      ],
      'pager_transform_links' => [
        'default' => 'relative',
      ],
      'pager_transform_links_destination' => [
        'default' => 'current',
      ],
    ] + parent::defineOptions();
  }

  /**
   * Get the options for row transform.
   *
   * @return array
   *   The options for row transform.
   */
  protected function getRowTransformOptions() {
    return [
      'entity' => $this->t('Entity'),
      'fields' => $this->t('Fields (Not Implemented)'),
    ];
  }

  /**
   * Get the options for pager transform.
   *
   * @return array
   *   The options for pager transform.
   */
  protected function getPagerTransformOptions() {
    return [
      'basic' => $this->t('Basic'),
      'mini' => $this->t('Mini'),
    ];
  }

  /**
   * Get the options for pager transform links.
   *
   * @return array
   *   The options for pager transform links.
   */
  protected function getPagerTransformLinksOptions() {
    return [
      'relative' => $this->t('Relative'),
      'absolute' => $this->t('Absolute'),
    ];
  }

  /**
   * Get the options for pager transform links destination.
   *
   * @return array
   *   The options for pager transform links destination.
   */
  protected function getPagerTransformLinksDestinationOptions() {
    return [
      'current' => $this->t('Current page'),
      'results' => $this->t('Results only page'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    if ($form_state->get('section') == 'row_transform') {
      $form['#title'] = $form['#title'] . ' ' . $this->t('How should a row be transformed?');
      $form['row_transform'] = [
        '#type' => 'radios',
        '#title' => $this->t('Row Transform'),
        '#description' => $this->t('Choose transform style for row result'),
        '#default_value' => $this->options['row_transform'],
        '#options' => $this->getRowTransformOptions(),
      ];
    }

    if ($form_state->get('section') == 'row_transform_mode') {
      $form['#title'] = $form['#title'] . ' ' . $this->t('Which transform mode should be used?');
      $form['row_transform_mode'] = [
        '#type' => 'radios',
        '#title' => $this->t('Row Transform Mode'),
        '#description' => $this->t('Choose transform style for row result'),
        '#default_value' => $this->options['row_transform_mode'],
        '#options' => $this->getRowTransformModes(),
      ];
    }

    if ($form_state->get('section') == 'pager_transform') {
      $form['#title'] = $form['#title'] . ' ' . $this->t('How should a pager be transformed?');
      $form['pager_transform'] = [
        '#type' => 'radios',
        '#title' => $this->t('Pager Transform'),
        '#description' => $this->t('Choose transform style for pager'),
        '#default_value' => $this->options['pager_transform'],
        '#options' => $this->getPagerTransformOptions(),
      ];
    }

    if ($form_state->get('section') == 'pager_transform_links') {
      $form['#title'] = $form['#title'] . ' ' . $this->t('How should a pager links be formatted?');
      $form['pager_transform_links'] = [
        '#type' => 'radios',
        '#title' => $this->t('Pager Transform Links'),
        '#description' => $this->t('Choose transform style for pager links'),
        '#default_value' => $this->options['pager_transform_links'],
        '#options' => $this->getPagerTransformLinksOptions(),
      ];
    }

    if ($form_state->get('section') == 'pager_transform_links_destination') {
      $form['#title'] = $form['#title'] . ' ' . $this->t('How should a pager links be formatted?');
      $form['pager_transform_links_destination'] = [
        '#type' => 'radios',
        '#title' => $this->t('Pager Transform Links Destination'),
        '#description' => $this->t('Choose transform style for pager links destination'),
        '#default_value' => $this->options['pager_transform_links_destination'],
        '#options' => $this->getPagerTransformLinksDestinationOptions(),
      ];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitOptionsForm(&$form, FormStateInterface $form_state) {
    if ($form_state->get('section') == 'row_transform') {
      $this->options['row_transform'] = $form_state->getValue('row_transform');
    }

    if ($form_state->get('section') == 'row_transform_mode') {
      $this->options['row_transform_mode'] = $form_state->getValue('row_transform_mode');
    }

    if ($form_state->get('section') == 'pager_transform') {
      $this->options['pager_transform'] = $form_state->getValue('pager_transform');
    }

    if ($form_state->get('section') == 'pager_transform_links') {
      $this->options['pager_transform_links'] = $form_state->getValue('pager_transform_links');
    }

    if ($form_state->get('section') == 'pager_transform_links_destination') {
      $this->options['pager_transform_links_destination'] = $form_state->getValue('pager_transform_links_destination');
    }
  }

  /**
   * {@inheritdoc}
   */
  public function optionsSummary(&$categories, &$options) {
    $categories['transform_api_views'] = [
      'title' => $this->t('Transform'),
      'column' => 'first',
    ];
    $options['row_transform'] = [
      'category' => 'transform_api_views',
      'title' => $this->t('Show'),
      'value' => $this->getRowTransformOptions()[$this->options['row_transform']],
    ];

    if ($this->options['row_transform'] == 'entity') {
      $options['row_transform_mode'] = [
        'category' => 'transform_api_views',
        'title' => $this->t('Mode'),
        'value' => $this->getRowTransformModes()[$this->options['row_transform_mode']],
      ];
    }

    $categories['transform_api_views_pager'] = [
      'title' => $this->t('Pager Transform'),
      'column' => 'second',
    ];

    $options['pager_transform'] = [
      'category' => 'transform_api_views_pager',
      'title' => $this->t('Pager transform'),
      'value' => $this->getPagerTransformOptions()[$this->options['pager_transform']],
    ];

    if ($this->options['pager_transform'] != 'basic') {
      $options['pager_transform_links'] = [
        'category' => 'transform_api_views_pager',
        'title' => $this->t('Pager transform links'),
        'value' => $this->getPagerTransformLinksOptions()[$this->options['pager_transform_links']],
      ];

      $options['pager_transform_links_destination'] = [
        'category' => 'transform_api_views_pager',
        'title' => $this->t('Pager transform links destination'),
        'value' => $this->getPagerTransformLinksDestinationOptions()[$this->options['pager_transform_links_destination']],
      ];
    }
  }

  /**
   * Get the options for row transform modes.
   *
   * @return array
   *   The options for row transform modes.
   */
  protected function getRowTransformModes() {
    if ($this->view->getQuery()->getPluginId() == 'search_api_query') {
      return $this->getSearchApiIndexTransformModes();
    }

    $entity_type = $this->view->getBaseEntityType();

    if ($entity_type == NULL) {
      return [
        'default' => $this->t('Default'),
      ];
    }

    return $this->entityTransformRepository->getTransformModeOptions($entity_type->id());
  }

  /**
   * Get the transform modes for entities in the search API index.
   *
   * @return array
   *   The transform modes for entities in the search API index.
   */
  protected function getSearchApiIndexTransformModes() {
    /** @var \Drupal\search_api\Entity\Index $index */
    $index = $this->view->getQuery()->getIndex();

    $entity_types = $index->getEntityTypes();

    $transform_modes = [
      'default' => $this->t('Default'),
    ];

    foreach ($entity_types as $entity_type) {
      $transform_modes = array_merge($transform_modes, $this->entityTransformRepository->getTransformModeOptions($entity_type));
    }

    return $transform_modes;
  }

}
