<?php

namespace Drupal\transform_api\Transform;

use Drupal\Core\Entity\EntityInterface;
use Drupal\transform_api\EntityTransformBuilderInterface;
use Drupal\transform_api\Repository\EntityTransformRepositoryInterface;

/**
 * A transform of one or more entities.
 */
class EntityTransform extends TransformBase {

  /**
   * The entities to transform.
   *
   * @var \Drupal\Core\Entity\EntityInterface[]
   */
  protected array $entities = [];

  /**
   * The transform mode to use for transformation.
   *
   * @var string
   */
  protected string $transformMode = EntityTransformRepositoryInterface::DEFAULT_DISPLAY_MODE;

  /**
   * The language to use for transformation.
   *
   * @var string|null
   */
  protected ?string $langcode = NULL;

  /**
   * Whether this transform is for multiple entities.
   *
   * @var bool
   */
  protected bool $multiple = FALSE;

  /**
   * Prepared transformation array for the entity.
   *
   * @var array
   */
  protected array $build = [];

  /**
   * Whether this transform should be cached or not.
   *
   * @var bool
   */
  protected bool $cacheable = TRUE;

  /**
   * Construct an EntityTransform.
   *
   * @param \Drupal\Core\Entity\EntityInterface|array $entity
   *   The entity or array of entities to be transformed.
   * @param string $transform_mode
   *   (Optional) The transform mode to use for transformation.
   * @param string $langcode
   *   (Optional) The language to use for transformation.
   */
  public function __construct(EntityInterface|array $entity, $transform_mode = EntityTransformRepositoryInterface::DEFAULT_DISPLAY_MODE, $langcode = NULL) {
    $this->transformMode = $transform_mode;
    $this->langcode = $langcode;
    if (is_array($entity)) {
      $this->setEntities($entity);
    }
    else {
      $this->setEntity($entity);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getTransformType() {
    return 'entity';
  }

  /**
   * {@inheritdoc}
   */
  public function getAlterIdentifiers() {
    return [$this->getTransformType(), $this->getEntity()->getEntityTypeId()];
  }

  /**
   * {@inheritdoc}
   */
  public function isMultiple() {
    return $this->multiple;
  }

  /**
   * {@inheritdoc}
   */
  public function shouldBeCached() {
    return $this->cacheable;
  }

  /**
   * Return the entity to be transformed.
   *
   * @return \Drupal\Core\Entity\EntityInterface
   *   The entity or NULL if not found.
   */
  public function getEntity(): EntityInterface {
    return reset($this->entities);
  }

  /**
   * Set the entity to be transformed.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to be transformed.
   * @param bool $configure
   *   Whether to configure the transform via EntityTransformBuilder.
   */
  public function setEntity(EntityInterface $entity, $configure = TRUE) {
    $this->multiple = FALSE;
    $this->entities = [$entity];
    if ($configure) {
      $this->getTransformBuilder()->configureTransform($this, $this->getEntity(), $this->getTransformMode(), $this->getLangcode());
    }
  }

  /**
   * Return the entities to be transformed.
   *
   * @return \Drupal\Core\Entity\EntityInterface[] The entities or an empty array if not found.
   *   The entities or an empty array if not found.
   */
  public function getEntities(): array {
    return $this->entities;
  }

  /**
   * Set the entities to be transformed.
   *
   * @param \Drupal\Core\Entity\EntityInterface[] $entities
   *   The entities to be transformed.
   */
  public function setEntities(array $entities) {
    $this->multiple = TRUE;
    $this->cacheable = FALSE;
    $this->entities = $entities;
  }

  /**
   * Return the transform mode used to be used for transformation.
   *
   * @return string
   *   The transform mode used to be used for transformation.
   */
  public function getTransformMode(): string {
    return $this->transformMode;
  }

  /**
   * Set the transform mode used to be used for transformation.
   *
   * @param string $transformMode
   *   The transform mode used to be used for transformation.
   */
  public function setTransformMode(string $transformMode) {
    $this->transformMode = $transformMode;
  }

  /**
   * Return the language code used to be used for transformation.
   *
   * @return string|null
   *   The language code used to be used for transformation.
   */
  public function getLangcode(): ?string {
    return $this->langcode;
  }

  /**
   * Set the language code used to be used for transformation.
   *
   * @param string $langcode
   *   The language code used to be used for transformation.
   */
  public function setLangcode(string $langcode) {
    $this->langcode = $langcode;
  }

  /**
   * Set the prepared transformation of the entity.
   *
   * @param array $build
   *   The prepared build.
   */
  public function setBuild(array $build) {
    $this->build = $build;
  }

  /**
   * Set whether the transformation should be cached.
   *
   * @param bool $cacheable
   *   Whether the transformation should be cached.
   */
  public function setCacheable(bool $cacheable): void {
    $this->cacheable = $cacheable;
  }

  /**
   * Get the EntityTransformBuilder for the entity's type.
   *
   * @return \Drupal\transform_api\EntityTransformBuilderInterface
   *   The EntityTransformBuilder for the entity's type.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   */
  protected function getTransformBuilder(): EntityTransformBuilderInterface {
    /** @var \Drupal\transform_api\EntityTransformBuilderInterface $handler */
    $handler = \Drupal::entityTypeManager()->getHandler($this->getEntity()->getEntityTypeId(), 'transform_builder');
    return $handler;
  }

  /**
   * {@inheritdoc}
   */
  public function transform(): array {
    if (!empty($this->build)) {
      $transformation = $this->build;
    }
    elseif (empty($this->entities)) {
      $transformation = [];
    }
    elseif ($this->isMultiple()) {
      $transformation = $this->getTransformBuilder()->transformMultiple($this->getEntities(), $this->getTransformMode(), $this->getLangcode());
    }
    else {
      $transformation = $this->getTransformBuilder()->transform($this->getEntity(), $this->getTransformMode(), $this->getLangcode());
    }
    $this->applyTo($transformation);
    return $transformation;
  }

  /**
   * Create an EntityTransform that is already prepared for transformation.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to be transformed.
   * @param array $build
   *   The prepared transformation of the entity.
   * @param string $transform_mode
   *   (Optional) The transform mode to be used for transformation.
   *
   * @return EntityTransform
   *   A fully prepared EntityTransform.
   *
   * @internal
   */
  public static function createPrepared(EntityInterface $entity, array $build, $transform_mode = EntityTransformRepositoryInterface::DEFAULT_DISPLAY_MODE): EntityTransform {
    $transform = new self($entity, $transform_mode);
    $transform->setBuild($build);
    return $transform;
  }

}
