<?php

namespace Drupal\transform_api_views\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\transform_api\Transformer;
use Drupal\transform_api_views\Transform\ViewTransform;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Controller for handling views transform routes.
 */
class TransformController extends ControllerBase {

  /**
   * The transformer service.
   *
   * @var \Drupal\transform_api\Transformer
   */
  protected Transformer $transformer;

  /**
   * The current request.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected Request $request;

  /**
   * Construct a TransformController object.
   *
   * @param \Drupal\transform_api\Transformer $transformer
   *   The transformer service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack service.
   */
  public function __construct(Transformer $transformer, RequestStack $requestStack) {
    $this->transformer = $transformer;
    $this->request = $requestStack->getCurrentRequest();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('transform_api.transformer'),
      $container->get('request_stack')
    );
  }

  /**
   * Transform a view's results.
   *
   * @param string $view_id
   *   The view ID.
   * @param string $display_id
   *   The display ID.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The transformed view results.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   * @throws \Drupal\transform_api\Exception\ResponseTransformationException
   */
  public function viewResults($view_id, $display_id): JsonResponse {
    $args = $this->request->query->all();

    $transform = new ViewTransform($view_id, $display_id, [], $args);

    return new JsonResponse($this->transformer->transformRoot($transform));
  }

}
