<?php

namespace Drupal\transform_api_views\Plugin\Transform\Type;

use Drupal\transform_api\Transform\EntityTransform;
use Drupal\transform_api\Transform\PagerTransform;
use Drupal\transform_api\Transform\TransformInterface;
use Drupal\transform_api\TransformationTypeBase;
use Drupal\transform_api_views\Transform\ViewsMiniPagerTransform;
use Drupal\transform_api_views\Transform\ViewTransform;
use Drupal\views\ResultRow;
use Drupal\views\ViewExecutable;
use Drupal\views\Views;

/**
 * Plugin for view transform types.
 *
 * @TransformationType(
 *  id = "view",
 *  title = "View transform"
 * )
 */
class View extends TransformationTypeBase {

  /**
   * {@inheritdoc}
   */
  public function transform(TransformInterface $transform) {
    /** @var \Drupal\transform_api_views\Transform\ViewTransform $view_transform */
    $view_transform = $transform;

    $view = $this->buildView($view_transform);

    $transformation = [
      'type' => 'view',
      'view_id' => $view_transform->getViewId(),
      'display_id' => $view_transform->getDisplayId(),
      '#args' => $view_transform->getArguments(),
      '#view' => $view,
    ];

    if ($view_transform->getOption('label_display') === 'visible') {
      $transformation['label'] = $view_transform->getOption('label');

      if ($transformation['label'] === '') {
        $transformation['label'] = $view->getTitle();
      }
    }

    $settings = $view->getDisplay()->getExtenders()['transform_api_views']->options;

    $row_transform = $settings['row_transform'];
    $row_transform_mode = $settings['row_transform_mode'];

    $transformation['results'] = [];

    foreach ($view->result as $row) {
      $transformation['results'][] = $this->buildRowTransform($row, $row_transform, $row_transform_mode);
    }

    $transformation['exposed_filters'] = $this->buildExposedFilters($view);
    $transformation['pager'] = $this->buildPagerTransform($view);

    // Cache for View transform. Including url.query_args context for filters.
    $transformation['#cache']['tags'] = $view->getCacheTags();
    $transformation['#cache']['contexts'] = $view->getDisplay()->getCacheMetadata()->getCacheContexts();

    return $transformation;
  }

  /**
   * Creates a ViewExecutable object from a ViewTransform object.
   *
   * @param \Drupal\transform_api_views\Transform\ViewTransform $view_transform
   *   The view transform object.
   *
   * @return \Drupal\views\ViewExecutable
   *   Created and executed view object.
   */
  protected function buildView(ViewTransform $view_transform): ViewExecutable {
    $view = Views::getView($view_transform->getViewId());

    $view->setDisplay($view_transform->getDisplayId());
    $view->setArguments($view_transform->getArguments());

    $items_per_page = $view_transform->getOption('items_per_page');

    if (!empty($items_per_page) && $items_per_page !== 'none') {
      $view->setItemsPerPage($items_per_page);
    }

    $view->preExecute();
    $view->execute();

    return $view;
  }

  /**
   * Transforms a row from a view.
   *
   * @param \Drupal\views\ResultRow $row
   *   The row to transform.
   * @param string $row_transform
   *   The type of transformation to apply to the row.
   * @param string $row_transform_mode
   *   The mode of the transformation.
   *
   * @return \Drupal\transform_api\Transform\TransformInterface
   *   The transformed row.
   *
   * @throws \Exception
   */
  protected function buildRowTransform(ResultRow $row, string $row_transform, string $row_transform_mode): TransformInterface {
    if ($row_transform == 'fields') {
      throw new \Exception('Fields transformation not implemented');
    }

    if ($row_transform == 'entity') {
      $entity = $row->_entity;

      return new EntityTransform($entity, $row_transform_mode);
    }

    throw new \InvalidArgumentException('Invalid row transformation type');
  }

  /**
   * Builds an array of exposed filters from a view.
   *
   * @param \Drupal\views\ViewExecutable $view
   *   The view object.
   *
   * @return array
   *   Array of exposed filters.
   */
  protected function buildExposedFilters(ViewExecutable $view): array {
    $exposed_filters = [];

    $raw_exposed_data = $view->exposed_raw_input;

    foreach ($raw_exposed_data as $filter_id => $filter_value) {
      $exposed_filters[$filter_id] = $this->buildExposedFilter($view, $filter_id);
    }

    return $exposed_filters;
  }

  /**
   * Builds an exposed filter from a view.
   *
   * @param \Drupal\views\ViewExecutable $view
   *   The view object.
   * @param string $filter_id
   *   The id of the exposed filter.
   *
   * @return array
   *   The exposed filter.
   */
  protected function buildExposedFilter(ViewExecutable $view, string $filter_id): array {
    $exposed_widget = $view->exposed_widgets[$filter_id] ?? NULL;

    if ($exposed_widget == NULL) {
      return [];
    }

    $type = $exposed_widget['#type'];

    $build = [
      'type' => $type,
      'label' => $this->getExposedFilterLabel($view, $filter_id),
      'value' => $exposed_widget['#value'],
    ];

    if ($type == 'select' || $type == 'radios') {
      $build['options'] = [];

      foreach ($exposed_widget['#options'] as $value => $label) {
        $build['options'][] = (object) [
          'value' => $value,
          'label' => $label,
        ];
      }
    }

    return $build;
  }

  /**
   * Get the label of an exposed filter.
   *
   * @param \Drupal\views\ViewExecutable $view
   *   The view object.
   * @param string $filter_id
   *   The id of the exposed filter.
   *
   * @return mixed|string
   *   The label of the exposed filter.
   */
  protected function getExposedFilterLabel(ViewExecutable $view, $filter_id) {
    // Workaround for better exposed filters.
    $is_bef = $view->exposed_widgets['#context']['bef'] ?? FALSE;

    if ($is_bef) {
      return $view->exposed_widgets[$filter_id]['#title'];
    }

    $filters = $view->exposed_widgets['#info'];

    foreach ($filters as $filter) {
      if ($filter['value'] == $filter_id) {
        return $filter['label'];
      }
    }

    return '';
  }

  /**
   * Builds a pager transform from a view.
   *
   * @param \Drupal\views\ViewExecutable $view
   *   The view object.
   *
   * @return \Drupal\transform_api\Transform\TransformInterface
   *   The pager transform.
   *
   * @throws \Exception
   */
  protected function buildPagerTransform(ViewExecutable $view): TransformInterface {
    $pager_transform = $view->getDisplay()
      ->getExtenders()['transform_api_views']->options['pager_transform'];

    switch ($pager_transform) {
      case 'basic':
        return new PagerTransform();

      case 'mini':
        return new ViewsMiniPagerTransform(0, $view);

      default:
        throw new \InvalidArgumentException('Invalid pager transform type');
    }
  }

}
