<?php

namespace Drupal\transform_api\Entity;

use Drupal\Component\Utility\Crypt;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Entity\EntityHandlerBase;
use Drupal\Core\Entity\EntityHandlerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Field\FieldItemInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Security\TrustedCallbackInterface;
use Drupal\Core\TypedData\TranslatableInterface as TranslatableDataInterface;
use Drupal\transform_api\Configs\EntityTransformDisplayInterface;
use Drupal\transform_api\EntityTransformBuilderInterface;
use Drupal\transform_api\EventSubscriber\TransformationCache;
use Drupal\transform_api\Repository\EntityTransformRepositoryInterface;
use Drupal\transform_api\Transform;
use Drupal\transform_api\Transform\EntityTransform;
use Drupal\transform_api\Transform\TransformInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for entity transform handlers.
 *
 * @ingroup entity_api
 */
class EntityTransformBuilder extends EntityHandlerBase implements EntityHandlerInterface, EntityTransformBuilderInterface, TrustedCallbackInterface {

  /**
   * The type of entities for which this transform handler is instantiated.
   *
   * @var string
   */
  protected $entityTypeId;

  /**
   * Information about the entity type.
   *
   * @var \Drupal\Core\Entity\EntityTypeInterface
   */
  protected $entityType;

  /**
   * The entity repository service.
   *
   * @var \Drupal\transform_api\Repository\EntityTransformRepositoryInterface
   */
  protected $entityRepository;

  /**
   * The entity display repository.
   *
   * @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface
   */
  protected $entityDisplayRepository;

  /**
   * The cache bin used to store the transformation cache.
   *
   * @var string
   */
  protected $cacheBin = 'transform';

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The transformation caching service.
   *
   * @var \Drupal\transform_api\EventSubscriber\TransformationCache
   */
  protected TransformationCache $transformationCache;

  /**
   * The EntityTransformDisplay objects created for individual field transforms.
   *
   * @var \Drupal\transform_api\Configs\EntityTransformDisplayInterface[]
   *
   * @see \Drupal\transform_api\Entity\EntityTransformBuilder::getSingleFieldDisplay()
   */
  protected $singleFieldDisplays;

  /**
   * Constructs a new EntityTransformBuilder.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type definition.
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   The entity repository service.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\transform_api\EventSubscriber\TransformationCache $transformation_cache
   *   The transformation cache.
   * @param \Drupal\transform_api\Repository\EntityTransformRepositoryInterface $entity_display_repository
   *   The entity display repository.
   */
  public function __construct(EntityTypeInterface $entity_type, EntityRepositoryInterface $entity_repository, LanguageManagerInterface $language_manager, TransformationCache $transformation_cache, EntityTransformRepositoryInterface $entity_display_repository) {
    $this->entityTypeId = $entity_type->id();
    $this->entityType = $entity_type;
    $this->entityRepository = $entity_repository;
    $this->languageManager = $language_manager;
    $this->transformationCache = $transformation_cache;
    $this->entityDisplayRepository = $entity_display_repository;
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
      $entity_type,
      $container->get('entity.repository'),
      $container->get('language_manager'),
      $container->get('transform_api.transformation_cache'),
      $container->get('transform_api.entity_display.repository')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function configureTransform(EntityTransform $transform, EntityInterface $entity, $transform_mode = 'full', $langcode = NULL) {
    // Allow modules to change the transform mode.
    $this->moduleHandler()->alter('entity_transform_mode', $transform_mode, $entity);

    // Ensure that from now on we are dealing with the proper translation
    // object.
    $entity = $this->entityRepository->getTranslationFromContext($entity, $langcode);
    $transform->setEntity($entity, FALSE);
    $transform->setValues([
      'entity_type' => $this->entityTypeId,
      'ids' => $entity->id(),
      'transform_mode' => $transform_mode,
    ]);

    // Cache the transformed output if permitted by the transform mode and
    // global entity type configuration.
    if ($this->isTransformModeCacheable($transform_mode) && !$entity->isNew() && $entity->isDefaultRevision() && $this->entityType->isRenderCacheable()) {
      $transform->addCacheTags(Cache::mergeTags($this->getCacheTags(), $entity->getCacheTags()));
      $transform->addCacheContexts($entity->getCacheContexts());
      $transform->setCacheMaxAge($entity->getCacheMaxAge());
      // @todo Add support for other cache bins.
      /* $transform->setCacheBin($this->cacheBin) */

      if ($entity instanceof TranslatableDataInterface && count($entity->getTranslationLanguages()) > 1) {
        $transform->setValue('langcode', $entity->language()->getId());
      }
    }
    else {
      $transform->setCacheable(FALSE);
    }

    // Allow modules to reconfigure.
    $this->moduleHandler()->invokeAll($entity->getEntityTypeId() . '_transform_configure', [
      $transform, $entity, $transform_mode, $langcode,
    ]);
    $this->moduleHandler()->invokeAll('entity_transform_configure', [$transform, $entity, $transform_mode, $langcode]);
  }

  /**
   * {@inheritdoc}
   */
  public function transform(EntityInterface $entity, $transform_mode = 'full', $langcode = NULL) {
    $build = $this->transformEntity($entity, $transform_mode, $langcode);
    $build['#pre_transform'][] = [$this, 'build'];

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks() {
    return ['build', 'buildMultiple'];
  }

  /**
   * {@inheritdoc}
   */
  public function transformMultiple(array $entities = [], $transform_mode = 'full', $langcode = NULL) {
    $build_list = [
      '#sorted' => TRUE,
      '#pre_transform' => [[$this, 'buildMultiple']],
    ];
    $weight = 0;
    foreach ($entities as $key => $entity) {
      $transform = new EntityTransform($entity, $transform_mode);
      $cached = $this->transformationCache->get($transform);
      if ($cached !== FALSE) {
        $transform->setBuild($cached);
        $transform->setWeight($weight++);
        $build_list[$key] = $transform;
      }
      else {
        $build_list[$key] = $this->transformEntity($transform->getEntity(), $transform_mode, $langcode);
        $build_list[$key]['#transform'] = $transform;
        $build_list[$key]['#weight'] = $weight++;
      }
    }

    return $build_list;
  }

  /**
   * Builds the transform array for an entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to transform.
   * @param string $transform_mode
   *   (optional) The transform mode that used to transform the entity.
   * @param string $langcode
   *   (optional) For which language the entity should be transformed, defaults
   *   to the current content language.
   *
   * @return array
   *   A transform array for the entity.
   */
  protected function transformEntity(EntityInterface $entity, $transform_mode = 'full', $langcode = NULL) {
    // Set build defaults.
    $transformation = $this->getBuildDefaults($entity, $transform_mode);
    $entityType = $this->entityTypeId;
    $this->moduleHandler()->alter([$entityType . '_transform_defaults', 'entity_transform_defaults'], $transformation, $entity, $transform_mode);
    return $transformation;
  }

  /**
   * Provides entity-specific defaults to the build process.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity for which the defaults should be provided.
   * @param string $transform_mode
   *   The transform mode that should be used.
   *
   * @return array
   *   Transformation array with defaults.
   */
  protected function getBuildDefaults(EntityInterface $entity, $transform_mode) {
    return [
      'type' => 'entity',
      'entity_type' => $entity->getEntityTypeId(),
      'bundle' => $entity->bundle(),
      'id' => $entity->id(),
      'transform_mode' => $transform_mode,
      "#{$this->entityTypeId}" => $entity,
      /*
       * @deprecated in transform_api:1.1 and is removed from transform_api:2.0.
       * Use #{entity_type_id} instead
       * @see https://www.drupal.org/node/3455892
       */
      "#entity" => $entity,
    ];
  }

  /**
   * Builds an entity's transform; augments entity defaults.
   *
   * This function is assigned as a #pre_transform callback in ::transform().
   *
   * It transforms the transformable array for a single entity to the same
   * structure as if we were transforming multiple entities, and then calls the
   * default ::buildMultiple() #pre_transform callback.
   *
   * @param array $build
   *   A transformable array containing build information and context for an
   *   entity transform.
   *
   * @return array
   *   The updated transformable array.
   *
   * @see \Drupal\Core\Render\RendererInterface::render()
   */
  public function build(array $build) {
    $build_list = [$build];
    $build_list = $this->buildMultiple($build_list);
    return $build_list[0];
  }

  /**
   * Builds multiple entities' transforms; augments entity defaults.
   *
   * This function is assigned as a #pre_transform callback
   * in ::transformMultiple().
   *
   * By delaying the building of an entity until the #pre_transform processing
   * in \Drupal::service('transform_api.transformer')->transform(), the
   * processing cost of assembling an entity's transformable array is saved
   * on cache-hit requests.
   *
   * @param array $build_list
   *   A transformable array containing build information and context for an
   *   entity transform.
   *
   * @return array
   *   The updated transformable array.
   *
   * @see \Drupal\Core\Render\RendererInterface::render()
   */
  public function buildMultiple(array $build_list) {
    // Build the transform modes and display objects.
    $transform_modes = [];
    $entity_type_key = "#{$this->entityTypeId}";
    $transform_hook = "{$this->entityTypeId}_transform";

    // Find the keys for the ContentEntities in the build; Store entities for
    // transformation by transform_mode.
    $children = Transform::children($build_list);
    foreach ($children as $key) {
      if ($build_list[$key] instanceof TransformInterface) {
        continue;
      }
      if (isset($build_list[$key][$entity_type_key])) {
        $entity = $build_list[$key][$entity_type_key];
        if ($entity instanceof FieldableEntityInterface) {
          $transform_modes[$build_list[$key]['transform_mode']][$key] = $entity;
        }
      }
    }

    // Build content for the displays represented by the entities.
    foreach ($transform_modes as $transform_mode => $transform_mode_entities) {
      $displays = EntityTransformDisplay::collectTransformDisplays($transform_mode_entities, $transform_mode);
      $this->buildComponents($build_list, $transform_mode_entities, $displays, $transform_mode);
      foreach (array_keys($transform_mode_entities) as $key) {
        // Allow for alterations while building, before transforming.
        $entity = $build_list[$key][$entity_type_key];
        $display = $displays[$entity->bundle()];

        $this->moduleHandler()->invokeAll($transform_hook, [&$build_list[$key], $entity, $display, $transform_mode]);
        $this->moduleHandler()->invokeAll('entity_transform', [&$build_list[$key], $entity, $display, $transform_mode]);

        $this->alterBuild($build_list[$key], $entity, $display, $transform_mode);
      }
    }

    foreach ($build_list as $key => $build) {
      if ($build instanceof TransformInterface) {
        continue;
      }
      if (isset($build['#transform'])) {
        /** @var \Drupal\transform_api\Transform\EntityTransform $transform */
        $transform = $build['#transform'];
        unset($build['#transform']);
        $transform->setBuild($build);
        $build_list[$key] = $transform;
      }
    }

    return $build_list;
  }

  /**
   * {@inheritdoc}
   */
  public function buildComponents(array &$build, array $entities, array $displays, $transform_mode) {
    $entities_by_bundle = [];
    foreach ($entities as $id => $entity) {
      // Initialize the field item attributes for the fields being displayed.
      // The entity can include fields that are not displayed, and the display
      // can include components that are not fields, so we want to act on the
      // intersection. However, the entity can have many more fields than are
      // displayed, so we avoid the cost of calling $entity->getProperties()
      // by iterating the intersection as follows.
      foreach ($displays[$entity->bundle()]->getComponents() as $name => $options) {
        if ($entity->hasField($name)) {
          foreach ($entity->get($name) as $item) {
            $item->_attributes = [];
          }
        }
      }

      // Group the entities by bundle.
      $entities_by_bundle[$entity->bundle()][$id] = $entity;
    }

    // Invoke hook_entity_prepare_transform().
    $this->moduleHandler()->invokeAll('entity_prepare_transform', [
      $this->entityTypeId,
      $entities,
      $displays,
      $transform_mode,
    ]);

    // Let the displays build their transform arrays.
    foreach ($entities_by_bundle as $bundle => $bundle_entities) {
      $display = $displays[$bundle];
      $display_build = $display->buildMultiple($bundle_entities);
      foreach ($bundle_entities as $id => $entity) {
        $build[$id] += $display_build[$id];

        if ($display->getComponent('label')) {
          $build[$id] += [
            'label' => $entity->label(),
          ];
        }

        if ($display->getComponent('url')) {
          $build[$id] += [
            'url' => $entity->toUrl()->toString(),
          ];
        }
      }
    }
  }

  /**
   * Specific per-entity building.
   *
   * @param array $build
   *   The transform array that is being created.
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to be prepared.
   * @param \Drupal\transform_api\Configs\EntityTransformDisplayInterface $display
   *   The entity transform display holding the display options configured for
   *   the entity components.
   * @param string $transform_mode
   *   The transform mode that should be used to prepare the entity.
   */
  protected function alterBuild(array &$build, EntityInterface $entity, EntityTransformDisplayInterface $display, $transform_mode) {}

  /**
   * {@inheritdoc}
   */
  public function getCacheTags() {
    return [$this->entityTypeId . '_transform'];
  }

  /**
   * {@inheritdoc}
   */
  public function resetCache(array $entities = NULL) {
    // If no set of specific entities is provided, invalidate the entity
    // transform builder's cache tag. This will invalidate all entities
    // transformed by this transform builder.
    // Otherwise, if a set of specific entities is provided, invalidate those
    // specific entities only, plus their list cache tags, because any lists in
    // which these entities are transformed, must be invalidated as well.
    // However, even in this case, we might invalidate more cache items than
    // necessary.
    // When we have a way to invalidate only those cache items that have both
    // the individual entity's cache tag and the transform builder's cache tag,
    // we'll be able to optimize this further.
    if (isset($entities)) {
      $tags = [];
      foreach ($entities as $entity) {
        $tags = Cache::mergeTags($tags, $entity->getCacheTags());
        $tags = Cache::mergeTags($tags, $entity->getEntityType()->getListCacheTags());
      }
      Cache::invalidateTags($tags);
    }
    else {
      Cache::invalidateTags($this->getCacheTags());
    }
  }

  /**
   * Determines whether the transform mode is cacheable.
   *
   * @param string $transform_mode
   *   Name of the transform mode that should be transformed.
   *
   * @return bool
   *   TRUE if the transform mode can be cached, FALSE otherwise.
   */
  protected function isTransformModeCacheable($transform_mode) {
    if ($transform_mode == 'default') {
      // The 'default' is not an actual transform mode.
      return TRUE;
    }
    $transform_modes_info = $this->entityDisplayRepository->getTransformModes($this->entityTypeId);
    return !empty($transform_modes_info[$transform_mode]['cache']);
  }

  /**
   * {@inheritdoc}
   */
  public function transformField(FieldItemListInterface $items, $display_options = []) {
    $entity = $items->getEntity();
    // If the field is not translatable and the entity is, then the field item
    // list always points to the default translation of the entity. Attempt to
    // fetch it in the current content language.
    if (!$items->getFieldDefinition()->isTranslatable() && $entity->isTranslatable()) {
      $entity = $this->entityRepository->getTranslationFromContext($entity);
    }

    $field_name = $items->getFieldDefinition()->getName();
    $display = $this->getSingleFieldDisplay($entity, $field_name, $display_options);

    $output = [];
    $build = $display->build($entity);
    if (isset($build[$field_name])) {
      $output = $build[$field_name];
    }

    return $output;
  }

  /**
   * {@inheritdoc}
   */
  public function transformFieldItem(FieldItemInterface $item, $display_options = []) {
    $entity = $item->getEntity();
    $field_name = $item->getFieldDefinition()->getName();

    // Clone the entity since we are going to modify field values.
    $clone = clone $entity;

    // Push the item as the single value for the field, and defer
    // to transformField() to build the transform array for the whole list.
    $clone->{$field_name}->setValue([$item->getValue()]);
    $elements = $this->transformField($clone->{$field_name}, $display_options);

    // Extract the part of the transform array we need.
    $output = $elements[0] ?? [];
    if (isset($elements['#access'])) {
      $output['#access'] = $elements['#access'];
    }

    return $output;
  }

  /**
   * Gets an EntityTransformDisplay for transforming an individual field.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   * @param string $field_name
   *   The field name.
   * @param string|array $display_options
   *   The display options passed to the transformField() method.
   *
   * @return \Drupal\transform_api\Configs\EntityTransformDisplayInterface
   *   An EntityTransformDisplay with a single field.
   */
  protected function getSingleFieldDisplay($entity, $field_name, $display_options) {
    if (is_string($display_options)) {
      // Transform mode: use the Display configured for the transform mode.
      $transform_mode = $display_options;
      $display = EntityTransformDisplay::collectTransformDisplay($entity, $transform_mode);
      // Hide all fields except the current one.
      foreach (array_keys($entity->getFieldDefinitions()) as $name) {
        if ($name != $field_name) {
          $display->removeComponent($name);
        }
      }
    }
    else {
      // Array of custom display options: use a runtime Display for the
      // '_custom' transform mode. Persist the displays created, to reduce
      // the number of objects (displays and transformer plugins) created
      // when transforming a series of fields individually for cases such
      // as views tables.
      $entity_type_id = $entity->getEntityTypeId();
      $bundle = $entity->bundle();
      $key = $entity_type_id . ':' . $bundle . ':' . $field_name . ':' . Crypt::hashBase64(serialize($display_options));
      if (!isset($this->singleFieldDisplays[$key])) {
        $this->singleFieldDisplays[$key] = EntityTransformDisplay::create([
          'targetEntityType' => $entity_type_id,
          'bundle' => $bundle,
          'status' => TRUE,
        ])->setComponent($field_name, $display_options);
      }
      $display = $this->singleFieldDisplays[$key];
    }

    return $display;
  }

}
