<?php

namespace Drupal\transform_api;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Field\FieldItemInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\transform_api\Transform\EntityTransform;

/**
 * Defines an interface for entity transform builders.
 *
 * @ingroup entity_api
 */
interface EntityTransformBuilderInterface {

  /**
   * Builds the component fields and properties of a set of entities.
   *
   * @param array &$build
   *   The transformable array representing the entity content.
   * @param \Drupal\Core\Entity\EntityInterface[] $entities
   *   The entities whose content is being built.
   * @param \Drupal\transform_api\Configs\EntityTransformDisplayInterface[] $displays
   *   The array of entity transform displays holding the display options
   *   configured for the entity components, keyed by bundle name.
   * @param string $transform_mode
   *   The transform mode in which the entity is being transformed.
   */
  public function buildComponents(array &$build, array $entities, array $displays, $transform_mode);

  /**
   * Configure a transform for an entity.
   *
   * @param \Drupal\transform_api\Transform\EntityTransform $transform
   *   The transform to configure.
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to configure the transform for.
   * @param string $transform_mode
   *   (optional) The transform mode that used to transform the entity.
   * @param string $langcode
   *   (optional) For which language the entity should be transformed, defaults
   *   to the current content language.
   */
  public function configureTransform(EntityTransform $transform, EntityInterface $entity, $transform_mode = 'full', $langcode = NULL);

  /**
   * Builds the transform array for the provided entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to transform.
   * @param string $transform_mode
   *   (optional) The transform mode that used to transform the entity.
   * @param string $langcode
   *   (optional) For which language the entity should be transformed, defaults
   *   to the current content language.
   *
   * @return array
   *   A transform array for the entity.
   *
   * @throws \InvalidArgumentException
   *   Can be thrown when the set of parameters is inconsistent, like when
   *   trying to transform a Comment and passing a Node which is not the one the
   *   comment belongs to, or not passing one, and having the comment node not
   *   be available for loading.
   */
  public function transform(EntityInterface $entity, $transform_mode = 'full', $langcode = NULL);

  /**
   * Builds the transform array for the provided entities.
   *
   * @param array $entities
   *   An array of entities implementing EntityInterface to transform.
   * @param string $transform_mode
   *   (optional) The transform mode that should be used to transform
   *   the entity.
   * @param string $langcode
   *   (optional) For which language the entity should be transformed, defaults
   *   to the current content language.
   *
   * @return array
   *   A transform array for the entities, indexed by the same keys as the
   *   entities array passed in $entities.
   *
   * @throws \InvalidArgumentException
   *   Can be thrown when the set of parameters is inconsistent, like when
   *   trying to transform Comments and passing a Node which is not the one the
   *   comments belongs to, or not passing one, and having the comments node not
   *   be available for loading.
   */
  public function transformMultiple(array $entities = [], $transform_mode = 'full', $langcode = NULL);

  /**
   * Resets the entity transform cache.
   *
   * @param \Drupal\Core\Entity\EntityInterface[] $entities
   *   (optional) If specified, the cache is reset for the given entities only.
   */
  public function resetCache(array $entities = NULL);

  /**
   * Builds a transformable array for the value of a single field in an entity.
   *
   * The resulting output is a fully themed field with label and multiple
   * values.
   *
   * This function can be used by third-party modules that need to output an
   * isolated field.
   * - The FieldTransformInterface::transform() method can be used to output a
   *   single formatted field value, without label or wrapping field
   *   transformation.
   *
   * The function takes care of invoking the prepare_transform steps. It also
   * respects field access permissions.
   *
   * @param \Drupal\Core\Field\FieldItemListInterface $items
   *   FieldItemList containing the values to be displayed.
   * @param string|array $display_options
   *   Can be either:
   *   - The name of a transform mode. The field will be displayed according to
   *     the display settings specified for this transform mode in the $field
   *     definition for the field in the entity's bundle. If no display settings
   *     are found for the transform mode, the settings for the 'default'
   *     transform mode will be used.
   *   - An array of display options. The following key/value pairs are allowed:
   *     - label: (string) Position of the label. The default 'field' theme
   *       implementation supports the values 'include' and 'omit'.
   *       Defaults to 'omit'.
   *     - type: (string) The transformer to use. Defaults to the
   *       'default_transformer' for the field type. The default transformer
   *       will also be used if the requested transformer is not available.
   *     - settings: (array) Settings specific to the transformer. Defaults to
   *       the transformer's default settings.
   *     - weight: (float) The weight to assign to the transformable element.
   *       Defaults to 0.
   *
   * @return array
   *   A transform array for the field values.
   *
   * @see \Drupal\Core\Entity\EntityTransformBuilderInterface::viewFieldItem()
   */
  public function transformField(FieldItemListInterface $items, $display_options = []);

  /**
   * Builds a transformable array for a single field item.
   *
   * @param \Drupal\Core\Field\FieldItemInterface $item
   *   FieldItem to be displayed.
   * @param string|array $display_options
   *   Can be either the name of a transform mode, or an array of display
   *   settings. See EntityTransformBuilderInterface::transformField() for
   *   more information.
   *
   * @return array
   *   A transform array for the field item.
   *
   * @see \Drupal\Core\Entity\EntityTransformBuilderInterface::viewField()
   */
  public function transformFieldItem(FieldItemInterface $item, $display_options = []);

  /**
   * The cache tag associated with this entity transform builder.
   *
   * An entity transform builder is instantiated on a per-entity type basis,
   * so the cache tags are also per-entity type.
   *
   * @return array
   *   An array of cache tags.
   */
  public function getCacheTags();

}
