<?php

namespace Drupal\transform_api\EventSubscriber;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\transform_api\Event\TransformMainContentEvent;
use Drupal\transform_api\Transform\EntityTransform;
use Drupal\transform_api\TransformEvents;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\Routing\Route;

/**
 * Handles canonical content entity urls as entity transforms.
 */
class EntityMainContentSubscriber implements EventSubscriberInterface {

  /**
   * The language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  private LanguageManagerInterface $languageManager;

  /**
   * Constructs a EntityMainContentSubscriber object.
   *
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager service.
   */
  public function __construct(LanguageManagerInterface $languageManager) {
    $this->languageManager = $languageManager;
  }

  /**
   * Handle entity routes as entity transforms.
   *
   * @param \Drupal\transform_api\Event\TransformMainContentEvent $event
   *   The Event to process.
   */
  public function onMainContent(TransformMainContentEvent $event) {
    $match = $event->getRouteMatch();
    $route_entity = $this->getEntityFromRouteMatch($match);
    if ($route_entity instanceof ContentEntityInterface &&
      (str_starts_with($match->getRouteName(), 'entity.')) &&
      (str_ends_with($match->getRouteName(), '.canonical'))) {
      $event->getCacheMetadata()->addCacheableDependency($route_entity);
      $langcode = $this->languageManager->getCurrentLanguage(LanguageInterface::TYPE_CONTENT)->getId();
      if ($route_entity->hasTranslation($langcode)) {
        $route_entity = $route_entity->getTranslation($langcode);
      }
      $event->setTransform(new EntityTransform($route_entity, 'full'));
    }
  }

  /**
   * Returns an entity parameter from a route match object.
   *
   * @return mixed|null
   *   The entity, or null if it's not an entity route.
   */
  protected function getEntityFromRouteMatch(RouteMatchInterface $match) {
    $route = $match->getRouteObject();
    if (!$route) {
      return NULL;
    }

    $entity_type_id = $this->getEntityTypeFromRoute($route);
    if ($entity_type_id) {
      return $match->getParameter($entity_type_id);
    }

    return NULL;
  }

  /**
   * Return the entity type id from a route object.
   *
   * @param \Symfony\Component\Routing\Route $route
   *   The route object.
   *
   * @return string|null
   *   The entity type id, null if it doesn't exist.
   */
  protected function getEntityTypeFromRoute(Route $route): ?string {
    if (!empty($route->getOptions()['parameters'])) {
      foreach ($route->getOptions()['parameters'] as $option) {
        if (isset($option['type']) && str_starts_with($option['type'], 'entity:')) {
          return substr($option['type'], strlen('entity:'));
        }
      }
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events[TransformEvents::MAIN_CONTENT][] = ['onMainContent'];
    return $events;
  }

}
