<?php

/**
 * @file
 * Hooks related to Transform API module.
 */

/**
 * @addtogroup hooks
 * @{
 */
use Drupal\transform_api\Transform\EntityTransform;

/**
 * Alter all transformation arrays after they are transformed.
 *
 * @param array $transformation
 *   The transformation array of the transform.
 */
function hook_transform_alter(&$transformation) {

}

/**
 * Alter transformation arrays with a specific TRANSFORM_TYPE identifier.
 *
 * These can include 'entity', 'field', 'block' and more.
 *
 * @param array $transformation
 *   The transformation array of the transform.
 */
function hook_TRANSFORM_TYPE_transform_alter(&$transformation) {

}

/**
 * Alter the configuration of transform blocks.
 *
 * @param array $blocks
 *   The array of transform blocks divided into regions.
 */
function hook_blocks_transform_config_alter(&$blocks) {

}

/**
 * Alter the transform array generated by an EntityDisplay for an entity.
 *
 * @param array $build
 *   The transform array generated by the EntityDisplay.
 * @param array $context
 *   An associative array containing:
 *   - entity: The entity being transformed.
 *   - transform_mode: The transform mode; for example, 'full' or 'teaser'.
 *   - display: The EntityDisplay holding the display options.
 *
 * @ingroup entity_crud
 */
function hook_entity_transform_build_alter(&$build, $context) {

}

/**
 * Change the transform mode of an entity that is being transformed.
 *
 * @param string $transform_mode
 *   The transform_mode that is to be used to transform the entity.
 * @param \Drupal\Core\Entity\EntityInterface $entity
 *   The entity that is being transformed.
 *
 * @ingroup entity_crud
 */
function hook_entity_transform_mode_alter(&$transform_mode, \Drupal\Core\Entity\EntityInterface $entity) {
  // For nodes, change the transform mode when it is teaser.
  if ($entity->getEntityTypeId() == 'node' && $transform_mode == 'teaser') {
    $transform_mode = 'my_custom_transform_mode';
  }
}

/**
 * Change the default configuration of an EntityTransform before transformation.
 *
 * This happens just before cache checking during transformation.
 *
 * Invoked for a specific entity type.
 *
 * @param \Drupal\transform_api\Transform\EntityTransform $transform
 *   The entity transform.
 * @param \Drupal\Core\Entity\EntityInterface $entity
 *   The entity to be transformed.
 * @param string $transform_mode
 *   The transform mode used for transformation.
 * @param string|null $langcode
 *   The language code used for transformation.
 *
 * @see hook_entity_transform_configure()
 */
function hook_ENTITY_TYPE_transform_configure(EntityTransform $transform, \Drupal\Core\Entity\EntityInterface $entity, $transform_mode, $langcode) {
}

/**
 * Change the default configuration of an EntityTransform before transformation.
 *
 * This happens just before cache checking during transformation.
 *
 * @param \Drupal\transform_api\Transform\EntityTransform $transform
 *   The entity transform.
 * @param \Drupal\Core\Entity\EntityInterface $entity
 *   The entity to be transformed.
 * @param string $transform_mode
 *   The transform mode used for transformation.
 * @param string|null $langcode
 *   The language code used for transformation.
 *
 * @see hook_ENTITY_TYPE_transform_configure()
 */
function hook_entity_transform_configure(EntityTransform $transform, \Drupal\Core\Entity\EntityInterface $entity, $transform_mode, $langcode) {
}

/**
 * Alter entity transformation array default values.
 *
 * Invoked for a specific entity type.
 *
 * @param array &$build
 *   A transform array containing the entity's default values.
 * @param \Drupal\Core\Entity\EntityInterface $entity
 *   The entity that is being transformed.
 * @param string $transform_mode
 *   The transform_mode that is to be used to transform the entity.
 *
 * @see \Drupal\Core\Render\RendererInterface::render()
 * @see \Drupal\Core\Entity\EntityViewBuilder
 * @see hook_entity_transform_defaults_alter()
 * @see hook_ENTITY_TYPE_transform_configure()
 *
 * @ingroup entity_crud
 */
function hook_ENTITY_TYPE_transform_defaults_alter(array &$build, \Drupal\Core\Entity\EntityInterface $entity, $transform_mode) {

}

/**
 * Alter entity transformation array default values.
 *
 * @param array &$build
 *   A transform array containing the entity's default values.
 * @param \Drupal\Core\Entity\EntityInterface $entity
 *   The entity that is being transformed.
 * @param string $transform_mode
 *   The transform_mode that is to be used to transform the entity.
 *
 * @see \Drupal\transform_api\Transformer::transform()
 * @see \Drupal\transform_api\Entity\EntityTransformBuilder
 * @see hook_ENTITY_TYPE_build_defaults_alter()
 * @see hook_entity_transform_configure()
 *
 * @ingroup entity_crud
 */
function hook_entity_transform_defaults_alter(array &$build, \Drupal\Core\Entity\EntityInterface $entity, $transform_mode) {

}

/**
 * Act on entities of a particular type being assembled before transforming.
 *
 * @param array &$build
 *   A transformable array representing the entity content. The module may add
 *   elements to $build prior to transforming. The structure of $build is a
 *   transformable array as expected by
 *   \Drupal\transform_api\TransformInterface::transform().
 * @param \Drupal\Core\Entity\EntityInterface $entity
 *   The entity object.
 * @param \Drupal\transform_api\Configs\EntityTransformDisplayInterface $display
 *   The entity transform display holding the display options configured for the
 *   entity components.
 * @param string $transform_mode
 *   The transform mode the entity is rendered in.
 *
 * @see hook_entity_transform_alter()
 * @see hook_ENTITY_TYPE_transform()
 *
 * @ingroup entity_crud
 */
function hook_entity_transform(array &$build, \Drupal\Core\Entity\EntityInterface $entity, \Drupal\transform_api\Configs\EntityTransformDisplayInterface $display, $transform_mode) {
  // Only do the extra work if the component is configured to be transformed.
  // This assumes a 'mymodule_addition' extra field has been defined for the
  // entity bundle in hook_entity_extra_field_info().
  if ($display->getComponent('mymodule_addition')) {
    $build['mymodule_addition'] = [
      'value' => mymodule_addition($entity),
      '#collapse' => TRUE,
    ];
  }
}

/**
 * Act on entities of a particular type being assembled before transforming.
 *
 * @param array &$build
 *   A transformable array representing the entity content. The module may add
 *   elements to $build prior to transforming. The structure of $build is a
 *   transformable array as expected by
 *   \Drupal\transform_api\TransformInterface::transform().
 * @param \Drupal\Core\Entity\EntityInterface $entity
 *   The entity object.
 * @param \Drupal\transform_api\Configs\EntityTransformDisplayInterface $display
 *   The entity transform display holding the display options configured for the
 *   entity components.
 * @param string $transform_mode
 *   The transform mode the entity is transformed in.
 *
 * @see hook_ENTITY_TYPE_transform_alter()
 * @see hook_entity_transform()
 *
 * @ingroup entity_crud
 */
function hook_ENTITY_TYPE_transform(array &$build, \Drupal\Core\Entity\EntityInterface $entity, \Drupal\transform_api\Configs\EntityTransformDisplayInterface $display, $transform_mode) {
  // Only do the extra work if the component is configured to be displayed.
  // This assumes a 'mymodule_addition' extra field has been defined for the
  // entity bundle in hook_entity_extra_field_info().
  if ($display->getComponent('mymodule_addition')) {
    $build['mymodule_addition'] = [
      'value' => mymodule_addition($entity),
      '#collapse' => TRUE,
    ];
  }
}

/**
 * Act on entities as they are being prepared for transformation.
 *
 * Allows you to operate on multiple entities as they are being prepared for
 * transformation. Only use this if attaching the data during the entity loading
 * phase is not appropriate, for example when attaching other 'entity' style
 * objects.
 *
 * @param string $entity_type_id
 *   The type of entities being transformed (i.e. node, user, comment).
 * @param array $entities
 *   The entities keyed by entity ID.
 * @param \Drupal\transform_api\Configs\EntityTransformDisplayInterface[] $displays
 *   The array of entity transform displays holding the display options
 *   configured for the entity components, keyed by bundle name.
 * @param string $transform_mode
 *   The transform mode.
 *
 * @ingroup entity_crud
 */
function hook_entity_prepare_transform($entity_type_id, array $entities, array $displays, $transform_mode) {
  // Load a specific node into the user object for later transformation.
  if (!empty($entities) && $entity_type_id == 'user') {
    // Only do the extra work if the component is configured to be
    // displayed. This assumes a 'mymodule_addition' extra field has been
    // defined for the entity bundle in hook_entity_extra_field_info().
    $ids = [];
    foreach ($entities as $id => $entity) {
      if ($displays[$entity->bundle()]->getComponent('mymodule_addition')) {
        $ids[] = $id;
      }
    }
    if ($ids) {
      $nodes = mymodule_get_user_nodes($ids);
      foreach ($ids as $id) {
        $entities[$id]->user_node = $nodes[$id];
      }
    }
  }
}

/**
 * @} End of "addtogroup hooks".
 */
