<?php

namespace Drupal\transform_api_facets\Plugin\Transform\Type;

use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\facets\Entity\Facet as FacetEntity;
use Drupal\facets\FacetManager\DefaultFacetManager;
use Drupal\transform_api\Transform\TransformInterface;
use Drupal\transform_api\TransformationTypeBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin for view transform types.
 *
 * @TransformationType(
 *  id = "facet",
 *  title = "Facet transform"
 * )
 */
class Facet extends TransformationTypeBase {

  /**
   * The facet manager.
   *
   * @var \Drupal\facets\FacetManager\DefaultFacetManager
   */
  protected DefaultFacetManager $facetManager;

  /**
   * The logger.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * The messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected MessengerInterface $messenger;

  /**
   * Constructs a new Facet object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\facets\FacetManager\DefaultFacetManager $facetManager
   *   The facet manager.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, DefaultFacetManager $facetManager, LoggerChannelInterface $logger, MessengerInterface $messenger) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->facetManager = $facetManager;
    $this->logger = $logger;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('facets.manager'),
      $container->get('logger.factory')->get('transform_api_facets'),
      $container->get('messenger')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function transform(TransformInterface $transform) {
    /** @var \Drupal\transform_api_facets\Transform\FacetTransform $facet_transform */
    $facet_transform = $transform;

    /** @var \Drupal\facets\Entity\Facet|null $facet */
    $facet = $this->facetManager->getEnabledFacets()[$facet_transform->getFacetId()] ?? NULL;

    if (!$facet) {
      return [];
    }

    $transformation = [
      'type' => 'facet',
      'facet_type' => $facet->getWidget()['type'],
      'facet_id' => $facet_transform->getFacetId(),
      'label' => $facet->label(),
      'url_alias' => $facet->getUrlAlias(),
      '#facet' => $facet,
      '#weight' => $facet->getWeight(),
    ];

    $this->buildFacet($facet, $transformation);

    $transformation['#cache'] = [
      'tags' => $facet->getCacheTags(),
      'contexts' => $facet->getCacheContexts(),
    ];

    return $transformation;
  }

  /**
   * Builds the facet transformation.
   *
   * @param \Drupal\facets\Entity\Facet $facet
   *   The facet.
   * @param array $transformation
   *   The transformation.
   *
   * @throws \Drupal\facets\Exception\InvalidProcessorException
   */
  protected function buildFacet(FacetEntity $facet, array &$transformation): void {
    $type = $facet->getWidget()['type'];

    $transformation['items'] = [];

    switch ($type) {
      case 'checkbox':
        $this->buildFacetFromCheckboxes($facet, $transformation);
        break;

      default:
        $message = $this->t('Facet list style not supported for facet @facet.', ['@facet' => $facet->label()]);
        $this->messenger->addWarning($message);
        $this->logger->error($message);
    }
  }

  /**
   * Builds the facet transformation for facets using 'checkbox' style.
   *
   * @param \Drupal\facets\Entity\Facet $facet
   *   The facet.
   * @param array $transformation
   *   The transformation.
   *
   * @throws \Drupal\facets\Exception\InvalidProcessorException
   */
  protected function buildFacetFromCheckboxes(FacetEntity $facet, &$transformation) {
    $build = $this->facetManager->build($facet)[0];

    if (isset($build['#type']) && $build['#type'] == 'container') {
      $build = $build[0];
    }

    // Replace the facet object with the one from the build.
    $facet = $build['#facet'];
    $transformation['#facet'] = $facet;

    $items = $build['#items'];

    foreach ($items as $item) {
      $value = $item['#title']['#raw_value'];

      $item_transform = [
        'type' => 'facet_item',
        'label' => $item['#title']['#value'],
        'count' => $item['#title']['#count'],
        'selected' => $facet->isActiveValue($value),
        'value' => $value,
      ];

      $transformation['items'][] = $item_transform;
    }
  }

}
