<?php

namespace Drupal\translation_owner\Drush\Commands;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Database\Connection;
use Drush\Commands\DrushCommands;
use Drush\Attributes as CLI;

/**
 * Drush commands for managing translation ownership.
 */
class TranslationOwnerCommands extends DrushCommands {

  protected $entityTypeManager;
  protected $database;

  /**
   * Constructs a TranslationOwnerCommands object.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    Connection $database
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->database = $database;
  }

  /**
   * Updates the owner (uid) of a node translation.
   */
  #[CLI\Command(name: 'translation-owner:update-uid', aliases: ['tou', 'translation-owner-update'])]
  public function updateTranslationUid(int $nid, string $langcode, int $new_uid) {
    try {
      $user_storage = $this->entityTypeManager->getStorage('user');
      $user = $user_storage->load($new_uid);
      if (!$user) {
        throw new \Exception(dt('User ID @uid does not exist.', ['@uid' => $new_uid]));
      }

      $node_storage = $this->entityTypeManager->getStorage('node');
      $node = $node_storage->load($nid);
      if (!$node) {
        throw new \Exception(dt('Node ID @nid does not exist.', ['@nid' => $nid]));
      }

      if (!$node->hasTranslation($langcode)) {
        throw new \Exception(dt('Translation @lang does not exist for node @nid.', [
          '@lang' => $langcode,
          '@nid' => $nid,
        ]));
      }

      $translation = $node->getTranslation($langcode);
      $old_uid = $translation->getOwnerId();

      $translation->setOwnerId($new_uid);
      
      // Wrap save operation to catch fatal errors from pathauto/path modules
      try {
        $translation->save();
        $this->logger()->success(dt('Successfully updated translation owner for node @nid (@lang) from user @old_uid to @new_uid', [
          '@nid' => $nid,
          '@lang' => $langcode,
          '@old_uid' => $old_uid,
          '@new_uid' => $new_uid,
        ]));
        return TRUE;
      }
      catch (\Throwable $save_error) {
        throw new \Exception(dt('Failed to save node @nid (@lang): @error', [
          '@nid' => $nid,
          '@lang' => $langcode,
          '@error' => $save_error->getMessage(),
        ]));
      }
    }
    catch (\Exception $e) {
      $this->logger()->error($e->getMessage());
      return FALSE;
    }
  }

  /**
   * Bulk updates translation owners based on a CSV file.
   */
  #[CLI\Command(name: 'translation-owner:bulk-update', aliases: ['toub', 'translation-owner-bulk'])]
  public function bulkUpdateTranslationOwners(string $file) {
    if (!file_exists($file)) {
      throw new \Exception(dt('File @file does not exist.', ['@file' => $file]));
    }

    $handle = fopen($file, 'r');
    $header = fgetcsv($handle);
    $required_columns = ['nid', 'langcode', 'new_uid'];

    if (array_diff($required_columns, $header)) {
      throw new \Exception(dt('CSV must contain columns: @columns', [
        '@columns' => implode(', ', $required_columns),
      ]));
    }

    $success_count = 0;
    $error_count = 0;
    $results = [];
    $line = 2;

    while (($data = fgetcsv($handle)) !== FALSE) {
      $row = array_combine($header, $data);
      $nid = $row['nid'];
      $langcode = $row['langcode'];
      $new_uid = $row['new_uid'];
      
      $result = $this->updateTranslationUid($nid, $langcode, $new_uid);
      
      if ($result === TRUE) {
        $success_count++;
        $status = 'SUCCESS';
      } else {
        $error_count++;
        $status = 'FAILED';
      }
      
      $results[] = [
        'nid' => $nid,
        'langcode' => $langcode,
        'new_uid' => $new_uid,
        'status' => $status,
      ];
      
      $line++;
    }
    fclose($handle);

    // Generate CSV report
    $report_file = dirname($file) . '/translation_owner_report_' . date('Y-m-d_H-i-s') . '.csv';
    $report_handle = fopen($report_file, 'w');
    
    // Write CSV header
    fputcsv($report_handle, ['nid', 'langcode', 'new_uid', 'status']);
    
    // Write results to CSV
    foreach ($results as $result) {
      fputcsv($report_handle, [
        $result['nid'],
        $result['langcode'],
        $result['new_uid'],
        $result['status']
      ]);
    }
    fclose($report_handle);
    
    // Display final report
    $this->logger()->success(dt('Bulk update completed with @success successes and @errors errors.', [
      '@success' => $success_count,
      '@errors' => $error_count,
    ]));
    
    $this->logger()->success(dt('Report saved to: @file', ['@file' => $report_file]));
    
    $this->output()->writeln("\nFinal Report:");
    $this->output()->writeln(sprintf("%-10s %-12s %-10s %-10s", 'NID', 'LANGCODE', 'NEW_UID', 'STATUS'));
    $this->output()->writeln(str_repeat('-', 46));
    
    foreach ($results as $result) {
      $this->output()->writeln(sprintf(
        "%-10s %-12s %-10s %-10s",
        $result['nid'],
        $result['langcode'],
        $result['new_uid'],
        $result['status']
      ));
    }
  }
}