<?php

namespace Drupal\translation_owner\Commands;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Database\Connection;
use Drush\Commands\DrushCommands;
use Drush\Attributes as CLI;

/**
 * Drush commands for managing translation ownership.
 */
class TranslationOwnerCommands extends DrushCommands {

  protected $entityTypeManager;
  protected $database;

  /**
   * Constructs a TranslationOwnerCommands object.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    Connection $database
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->database = $database;
  }

  /**
   * Updates the owner (uid) of a node translation.
   */
  #[CLI\Command(name: 'translation-owner:update-uid', aliases: ['tou', 'translation-owner-update'])]
  public function updateTranslationUid(int $nid, string $langcode, int $new_uid) {
    try {
      $user_storage = $this->entityTypeManager->getStorage('user');
      $user = $user_storage->load($new_uid);
      if (!$user) {
        throw new \Exception(dt('User ID @uid does not exist.', ['@uid' => $new_uid]));
      }

      $node_storage = $this->entityTypeManager->getStorage('node');
      
      // Clear entity cache to prevent stale data
      $node_storage->resetCache([$nid]);
      
      $node = $node_storage->load($nid);
      if (!$node) {
        throw new \Exception(dt('Node ID @nid does not exist.', ['@nid' => $nid]));
      }

      if (!$node->hasTranslation($langcode)) {
        throw new \Exception(dt('Translation @lang does not exist for node @nid.', [
          '@lang' => $langcode,
          '@nid' => $nid,
        ]));
      }

      $translation = $node->getTranslation($langcode);
      $old_uid = $translation->getOwnerId();

      $translation->setOwnerId($new_uid);
      $translation->save();
      
      // Clear cache after save to prevent cross-contamination
      $node_storage->resetCache([$nid]);

      $this->logger()->success(dt('Successfully updated translation owner for node @nid (@lang) from user @old_uid to @new_uid', [
        '@nid' => $nid,
        '@lang' => $langcode,
        '@old_uid' => $old_uid,
        '@new_uid' => $new_uid,
      ]));
    }
    catch (\Exception $e) {
      $this->logger()->error($e->getMessage());
      return 1;
    }
  }

  /**
   * Bulk updates translation owners based on a CSV file.
   */
  #[CLI\Command(name: 'translation-owner:bulk-update', aliases: ['toub', 'translation-owner-bulk'])]
  public function bulkUpdateTranslationOwners(string $file) {
    if (!file_exists($file)) {
      throw new \Exception(dt('File @file does not exist.', ['@file' => $file]));
    }

    $handle = fopen($file, 'r');
    $header = fgetcsv($handle);
    $required_columns = ['nid', 'langcode', 'new_uid'];

    if (array_diff($required_columns, $header)) {
      throw new \Exception(dt('CSV must contain columns: @columns', [
        '@columns' => implode(', ', $required_columns),
      ]));
    }

    $success_count = 0;
    $error_count = 0;
    $line = 2;

    while (($data = fgetcsv($handle)) !== FALSE) {
      $row = array_combine($header, $data);
      try {
        $this->updateTranslationUid(
          $row['nid'],
          $row['langcode'],
          $row['new_uid']
        );
        $success_count++;
      }
      catch (\Exception $e) {
        $this->logger()->warning(dt('Error on line @line: @message', [
          '@line' => $line,
          '@message' => $e->getMessage(),
        ]));
        $error_count++;
      }
      $line++;
    }
    fclose($handle);

    $this->logger()->success(dt('Bulk update completed with @success successes and @errors errors.', [
      '@success' => $success_count,
      '@errors' => $error_count,
    ]));
  }
}
