<?php

namespace Drupal\translation_owner\Drush\Commands;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Database\Connection;
use Drush\Commands\DrushCommands;
use Drush\Attributes as CLI;

/**
 * Drush commands for managing translation ownership.
 */
class TranslationOwnerCommands extends DrushCommands {

  protected $entityTypeManager;
  protected $database;

  /**
   * Constructs a TranslationOwnerCommands object.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    Connection $database
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->database = $database;
  }

  /**
   * Updates the owner (uid) of a node translation.
   */
  #[CLI\Command(name: 'translation-owner:update-uid', aliases: ['tou', 'translation-owner-update'])]
  public function updateTranslationUid(int $nid, string $langcode, int $new_uid) {
    try {
      $user_storage = $this->entityTypeManager->getStorage('user');
      $user = $user_storage->load($new_uid);
      if (!$user) {
        throw new \Exception(dt('User ID @uid does not exist.', ['@uid' => $new_uid]));
      }

      $node_storage = $this->entityTypeManager->getStorage('node');
      
      // Clear entity cache to prevent stale data
      $node_storage->resetCache([$nid]);
      
      $node = $node_storage->load($nid);
      if (!$node) {
        throw new \Exception(dt('Node ID @nid does not exist.', ['@nid' => $nid]));
      }

      if (!$node->hasTranslation($langcode)) {
        throw new \Exception(dt('Translation @lang does not exist for node @nid.', [
          '@lang' => $langcode,
          '@nid' => $nid,
        ]));
      }

      // Get current UID from database directly to avoid cached values
      $current_uid = $this->database->select('node_field_data', 'nfd')
        ->fields('nfd', ['uid'])
        ->condition('nid', $nid)
        ->condition('langcode', $langcode)
        ->execute()
        ->fetchField();

      // Only update if the UID is actually different
      if ($current_uid != $new_uid) {
        // Get current revision ID
        $current_vid = $this->database->select('node_field_data', 'nfd')
          ->fields('nfd', ['vid'])
          ->condition('nid', $nid)
          ->condition('langcode', $langcode)
          ->execute()
          ->fetchField();
        
        // Update database directly to avoid affecting other translations
        $updated_data = $this->database->update('node_field_data')
          ->fields(['uid' => $new_uid])
          ->condition('nid', $nid)
          ->condition('langcode', $langcode)
          ->execute();
          
        $updated_revision = $this->database->update('node_field_revision')
          ->fields(['uid' => $new_uid])
          ->condition('nid', $nid)
          ->condition('langcode', $langcode)
          ->condition('vid', $current_vid)
          ->execute();
        
        if ($updated_data || $updated_revision) {
          // Comprehensive cache invalidation to force entity reloading
          $node_storage->resetCache([$nid]);
          \Drupal::service('cache_tags.invalidator')->invalidateTags(['node:' . $nid]);
          \Drupal::service('entity.memory_cache')->deleteAll();
          \Drupal::cache('entity')->delete('values:node:' . $nid);
          
          $this->logger()->success(dt('Successfully updated translation owner for node @nid (@lang) from user @old_uid to @new_uid', [
            '@nid' => $nid,
            '@lang' => $langcode,
            '@old_uid' => $current_uid,
            '@new_uid' => $new_uid,
          ]));
          return TRUE;
        } else {
          throw new \Exception(dt('Failed to update database for node @nid (@lang)', [
            '@nid' => $nid,
            '@lang' => $langcode,
          ]));
        }
      } else {
        // No update needed - UID already correct
        $this->logger()->info(dt('No update needed for node @nid (@lang) - already has UID @uid', [
          '@nid' => $nid,
          '@lang' => $langcode,
          '@uid' => $new_uid,
        ]));
        return TRUE;
      }
    }
    catch (\Exception $e) {
      $this->logger()->error($e->getMessage());
      return FALSE;
    }
  }

  /**
   * Bulk updates translation owners based on a CSV file.
   */
  #[CLI\Command(name: 'translation-owner:bulk-update', aliases: ['toub', 'translation-owner-bulk'])]
  public function bulkUpdateTranslationOwners(string $file) {
    if (!file_exists($file)) {
      throw new \Exception(dt('File @file does not exist.', ['@file' => $file]));
    }

    $handle = fopen($file, 'r');
    $header = fgetcsv($handle);
    $required_columns = ['nid', 'langcode', 'new_uid'];

    if (array_diff($required_columns, $header)) {
      throw new \Exception(dt('CSV must contain columns: @columns', [
        '@columns' => implode(', ', $required_columns),
      ]));
    }

    $success_count = 0;
    $error_count = 0;
    $results = [];
    $line = 2;

    while (($data = fgetcsv($handle)) !== FALSE) {
      $row = array_combine($header, $data);
      $nid = $row['nid'];
      $langcode = $row['langcode'];
      $new_uid = $row['new_uid'];
      
      $result = $this->updateTranslationUid($nid, $langcode, $new_uid);
      
      if ($result === TRUE) {
        $success_count++;
        $status = 'SUCCESS';
      } else {
        $error_count++;
        $status = 'FAILED';
      }
      
      $results[] = [
        'nid' => $nid,
        'langcode' => $langcode,
        'new_uid' => $new_uid,
        'status' => $status,
      ];
      
      $line++;
    }
    fclose($handle);

    // Generate CSV report
    $report_file = dirname($file) . '/translation_owner_report_' . date('Y-m-d_H-i-s') . '.csv';
    $report_handle = fopen($report_file, 'w');
    
    if ($report_handle !== FALSE) {
      // Write CSV header
      fputcsv($report_handle, ['nid', 'langcode', 'new_uid', 'status']);
      
      // Write results to CSV
      foreach ($results as $result) {
        fputcsv($report_handle, [
          $result['nid'],
          $result['langcode'],
          $result['new_uid'],
          $result['status']
        ]);
      }
      fclose($report_handle);
    } else {
      $this->logger()->warning(dt('Could not create report file: @file', ['@file' => $report_file]));
      $report_file = NULL;
    }
    
    // Display final report
    $this->logger()->success(dt('Bulk update completed with @success successes and @errors errors.', [
      '@success' => $success_count,
      '@errors' => $error_count,
    ]));
    
    if ($report_file) {
      $this->logger()->success(dt('Report saved to: @file', ['@file' => $report_file]));
    }
    
    $this->output()->writeln("\nFinal Report:");
    $this->output()->writeln(sprintf("%-10s %-12s %-10s %-10s", 'NID', 'LANGCODE', 'NEW_UID', 'STATUS'));
    $this->output()->writeln(str_repeat('-', 46));
    
    foreach ($results as $result) {
      $this->output()->writeln(sprintf(
        "%-10s %-12s %-10s %-10s",
        $result['nid'],
        $result['langcode'],
        $result['new_uid'],
        $result['status']
      ));
    }
  }
}