<?php

namespace Drupal\transparse\Commands;

use Drupal\transparse\Service\TranslationParser;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Drush commands for the Transparse module.
 */
final class TransparseCommands extends DrushCommands {

  /**
   * The translation parser service.
   *
   * @var \Drupal\transparse\Service\TranslationParser
   */
  protected TranslationParser $translationParser;

  /**
   * Constructs a TransparseCommands object.
   *
   * @param \Drupal\transparse\Service\TranslationParser $translation_parser
   *   The translation parser service.
   */
  public function __construct(TranslationParser $translation_parser) {
    parent::__construct();
    $this->translationParser = $translation_parser;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('transparse.translation_parser')
    );
  }

  /**
   * Scans the active theme for translatable strings and registers them.
   *
   * @param array $options
   *   The command options.
   *
   * @command transparse:scan
   * @option force rescan, ignoring cached results
   * @option show-strings Show all found strings in output
   * @option path Specific path to scan (defaults to active theme)
   * @aliases tps, transparse-scan
   * @usage transparse:scan
   *   Scan the active theme for translatable strings
   * @usage transparse:scan --force
   *   Force a fresh scan, ignoring cache
   * @usage transparse:scan --show-strings
   *   Show all found strings in detailed output
   * @usage transparse:scan --path=web/themes/custom/mytheme
   *   Scan a specific directory
   */
  #[CLI\Command(name: 'transparse:scan', aliases: ['tps', 'transparse-scan'])]
  #[CLI\Option(name: 'force', description: 'Force rescan, ignoring cached results')]
  #[CLI\Option(name: 'show-strings', description: 'Show all found strings in output')]
  #[CLI\Option(name: 'path', description: 'Specific path to scan')]
  #[CLI\Usage(name: 'transparse:scan', description: 'Scan the active theme for translatable strings')]
  #[CLI\Usage(name: 'transparse:scan --force', description: 'Force a fresh scan, ignoring cache')]
  #[CLI\Usage(name: 'transparse:scan --show-strings', description: 'Show all found strings in detailed output')]
  public function scan(array $options = ['force' => FALSE, 'show-strings' => FALSE, 'path' => NULL]): void {
    $this->output()->writeln('Starting translation string scan...');

    // Clear cache if forced.
    if ($options['force']) {
      $this->translationParser->clearCache();
      $this->output()->writeln('Cache cleared. Performing fresh scan...');
    }

    // Scan for strings.
    $strings = $this->translationParser->scanTheme($options['path'], $options['force']);

    // Count strings by type.
    $counts = [
      'field_description' => 0,
      'field_group_label' => 0,
      'field_label' => 0,
      'field_placeholder' => 0,
      'field_storage_allowed_value' => 0,
      'trans_block' => 0,
      'twig_alt' => 0,
      'twig_filter' => 0,
      'twig_title' => 0,
    ];

    foreach ($strings as $string_data) {
      if (isset($counts[$string_data['type']])) {
        $counts[$string_data['type']]++;
      }
    }

    // Register strings with translation system.
    $registered = $this->translationParser->registerStrings($strings);

    // Output results.
    $this->output()->writeln('');
    $this->output()->writeln('<info>Scan complete!</info>');
    $this->output()->writeln('');
    $this->output()->writeln('Summary:');
    $this->output()->writeln('  Total strings found: ' . count($strings));
    $this->output()->writeln('  Strings registered: ' . $registered);
    $this->output()->writeln('');
    $this->output()->writeln('Breakdown by type:');
    $this->output()->writeln('  Twig |t filters: ' . $counts['twig_filter']);
    $this->output()->writeln('  {% trans %} blocks: ' . $counts['trans_block']);
    $this->output()->writeln('  Twig alt attributes: ' . $counts['twig_alt']);
    $this->output()->writeln('  Twig title attributes: ' . $counts['twig_title']);
    $this->output()->writeln('  Field labels: ' . $counts['field_label']);
    $this->output()->writeln('  Field descriptions: ' . $counts['field_description']);
    $this->output()->writeln('  Field placeholders: ' . $counts['field_placeholder']);
    $this->output()->writeln('  Field group labels: ' . $counts['field_group_label']);
    $this->output()->writeln('  Field storage allowed values: ' . $counts['field_storage_allowed_value']);
    $this->output()->writeln('');

    // Show detailed output if show-strings is enabled.
    if ($options['show-strings']) {
      $this->output()->writeln('<comment>Found strings:</comment>');
      $this->output()->writeln('');

      $grouped = [];
      foreach ($strings as $string_data) {
        $grouped[$string_data['type']][] = $string_data;
      }

      foreach ($grouped as $type => $type_strings) {
        $this->output()->writeln('<info>' . ucfirst(str_replace('_', ' ', $type)) . ':</info>');
        foreach ($type_strings as $string_data) {
          $this->output()->writeln('  - "' . $string_data['string'] . '" [' . $string_data['source'] . ']');
        }
        $this->output()->writeln('');
      }
    }

    $this->output()->writeln('<info>Strings are now available in the Translation UI:</info>');
    $this->output()->writeln('/admin/config/regional/translate');
  }

  /**
   * Clears the transparse translation string cache.
   *
   * @command transparse:clear-cache
   * @aliases tpc, transparse-clear
   * @usage transparse:clear-cache
   *   Clear the cached translation strings
   */
  #[CLI\Command(name: 'transparse:clear-cache', aliases: ['tpc', 'transparse-clear'])]
  #[CLI\Usage(name: 'transparse:clear-cache', description: 'Clear the cached translation strings')]
  public function clearCache(): void {
    $this->translationParser->clearCache();
    $this->output()->writeln('<info>Translation string cache cleared successfully.</info>');
    $this->output()->writeln('Run <comment>drush transparse:scan</comment> to rescan.');
  }

}
