<?php

namespace Drupal\transparse\Service;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ThemeHandlerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Site\Settings;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\locale\StringStorageInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Service to parse and extract translatable strings from theme files.
 */
class TranslationParser {

  use StringTranslationTrait;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected FileSystemInterface $fileSystem;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The theme handler.
   *
   * @var \Drupal\Core\Extension\ThemeHandlerInterface
   */
  protected ThemeHandlerInterface $themeHandler;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected CacheBackendInterface $cache;

  /**
   * The locale string storage.
   *
   * @var \Drupal\locale\StringStorageInterface
   */
  protected StringStorageInterface $localeStorage;

  /**
   * Constructs a TranslationParser object.
   *
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Extension\ThemeHandlerInterface $theme_handler
   *   The theme handler.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation service.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   * @param \Drupal\locale\StringStorageInterface $locale_storage
   *   The locale string storage.
   */
  public function __construct(
    FileSystemInterface $file_system,
    ConfigFactoryInterface $config_factory,
    ThemeHandlerInterface $theme_handler,
    LanguageManagerInterface $language_manager,
    TranslationInterface $string_translation,
    CacheBackendInterface $cache,
    StringStorageInterface $locale_storage,
  ) {
    $this->fileSystem = $file_system;
    $this->configFactory = $config_factory;
    $this->themeHandler = $theme_handler;
    $this->languageManager = $language_manager;
    $this->stringTranslation = $string_translation;
    $this->cache = $cache;
    $this->localeStorage = $locale_storage;
  }

  /**
   * Scans the active theme for translatable strings.
   *
   * @param string|null $path
   *   Optional specific path to scan.
   * @param bool $force
   *   Force rescan, ignoring cache.
   *
   * @return array
   *   Array of found translatable strings with metadata.
   */
  public function scanTheme(?string $path = NULL, bool $force = FALSE): array {
    $cache_id = 'transparse:strings';
    $default_theme = $this->configFactory->get('system.theme')->get('default');

    // Return cached data if available and not forcing rescan.
    if (!$force && $cache = $this->cache->get($cache_id)) {
      return $cache->data;
    }

    $strings = [];

    // Scan Twig templates.
    if ($path) {
      $theme_path = $path;
    }
    else {
      // Get the default theme's path dynamically.
      $theme = $this->themeHandler->getTheme($default_theme);
      $theme_path = DRUPAL_ROOT . '/' . $theme->getPath();
    }
    $twig_strings = $this->scanTwigFiles($theme_path);
    $strings = array_merge($strings, $twig_strings);

    // Scan field configurations for all entity types.
    $field_strings = $this->scanFieldConfigs();
    $strings = array_merge($strings, $field_strings);

    // Scan field_group configurations.
    $field_group_strings = $this->scanFieldGroupConfigs();
    $strings = array_merge($strings, $field_group_strings);

    // Remove duplicates.
    $strings = array_unique($strings, SORT_REGULAR);

    // Cache the results.
    $this->cache->set($cache_id, $strings, CacheBackendInterface::CACHE_PERMANENT, ['transparse']);

    return $strings;
  }

  /**
   * Scans Twig template files for translatable strings.
   *
   * @param string $directory
   *   The directory to scan.
   *
   * @return array
   *   Array of found strings with metadata.
   */
  protected function scanTwigFiles(string $directory): array {
    $strings = [];

    // Get all Twig files.
    $files = $this->findTwigFiles($directory);

    foreach ($files as $file) {
      $content = file_get_contents($file);
      if ($content === FALSE) {
        continue;
      }

      // Extract strings using |t filter: {{ 'text'|t }}.
      preg_match_all("/\{\{\s*['\"]([^'\"]+)['\"]\s*\|\s*t\s*\}\}/", $content, $matches);
      if (!empty($matches[1])) {
        foreach ($matches[1] as $string) {
          $strings[] = [
            'string' => $string,
            'source' => basename($file),
            'type' => 'twig_filter',
          ];
        }
      }

      // Extract strings from {% trans %} blocks.
      preg_match_all("/\{\%\s*trans\s*\%\}(.*?)\{\%\s*endtrans\s*\%\}/s", $content, $trans_matches);
      if (!empty($trans_matches[1])) {
        foreach ($trans_matches[1] as $string) {
          $clean_string = trim(strip_tags($string));
          if (!empty($clean_string)) {
            $strings[] = [
              'string' => $clean_string,
              'source' => basename($file),
              'type' => 'trans_block',
            ];
          }
        }
      }

      // Extract strings with alt attribute patterns: alt={{"text"|t}}.
      preg_match_all('/alt\s*=\s*\{\{\s*["\']([^"\']+)["\']\s*\|\s*t\s*\}\}/', $content, $alt_matches);
      if (!empty($alt_matches[1])) {
        foreach ($alt_matches[1] as $string) {
          $strings[] = [
            'string' => $string,
            'source' => basename($file),
            'type' => 'twig_alt',
          ];
        }
      }

      // Extract strings with title attribute patterns: title="{{ 'text'|t }}".
      preg_match_all('/title\s*=\s*["\']?\{\{\s*["\']([^"\']+)["\']\s*\|\s*t\s*\}\}["\']?/', $content, $title_matches);
      if (!empty($title_matches[1])) {
        foreach ($title_matches[1] as $string) {
          $strings[] = [
            'string' => $string,
            'source' => basename($file),
            'type' => 'twig_title',
          ];
        }
      }
    }

    return $strings;
  }

  /**
   * Finds all Twig files in a directory recursively.
   *
   * @param string $directory
   *   The directory to search.
   *
   * @return array
   *   Array of file paths.
   */
  protected function findTwigFiles(string $directory): array {
    $files = [];

    if (!is_dir($directory)) {
      return $files;
    }

    $iterator = new \RecursiveIteratorIterator(
      new \RecursiveDirectoryIterator($directory, \FilesystemIterator::SKIP_DOTS),
      \RecursiveIteratorIterator::SELF_FIRST
    );

    foreach ($iterator as $file) {
      if ($file->isFile() && $file->getExtension() === 'twig') {
        // Include all Twig templates (not just node-related).
        $files[] = $file->getPathname();
      }
    }

    return $files;
  }

  /**
   * Scans field configuration files for labels and descriptions.
   *
   * Scans all entity types: node, block_content, group, taxonomy_term,
   * paragraph, and field_storage_config files.
   *
   * @return array
   *   Array of found strings with metadata.
   */
  protected function scanFieldConfigs(): array {
    $strings = [];

    // Get config path from settings.
    // Note: Settings::get('config_sync_directory') returns an absolute path.
    $config_path = Settings::get('config_sync_directory');

    if (!is_dir($config_path)) {
      return $strings;
    }

    // Common Entity types with labels to scan for field configurations.
    $entity_types = [
      'block_content',
      'group',
      'node',
      'paragraph',
      'taxonomy_term',
    ];

    // Scan field.field.{entity_type}.*.yml files for all entity types.
    foreach ($entity_types as $entity_type) {
      $pattern = $config_path . '/field.field.' . $entity_type . '.*.yml';
      $files = glob($pattern);

      foreach ($files as $file) {
        $config = Yaml::parseFile($file);

        if (!empty($config['label'])) {
          $strings[] = [
            'string' => $config['label'],
            'source' => basename($file),
            'type' => 'field_label',
            'entity_type' => $entity_type,
          ];
        }

        if (!empty($config['description'])) {
          $strings[] = [
            'string' => $config['description'],
            'source' => basename($file),
            'type' => 'field_description',
            'entity_type' => $entity_type,
          ];
        }
      }
    }

    // Scan field.storage.*.yml files for field storage labels.
    $pattern = $config_path . '/field.storage.*.yml';
    $files = glob($pattern);

    foreach ($files as $file) {
      $config = Yaml::parseFile($file);

      if (!empty($config['settings']['allowed_values'])) {
        // Extract allowed values labels (for list fields).
        foreach ($config['settings']['allowed_values'] as $value) {
          if (!empty($value['label'])) {
            $strings[] = [
              'string' => $value['label'],
              'source' => basename($file),
              'type' => 'field_storage_allowed_value',
            ];
          }
        }
      }
    }

    // Scan core.entity_form_display.*.yml for placeholders (all entity types).
    foreach ($entity_types as $entity_type) {
      $pattern = $config_path . '/core.entity_form_display.' . $entity_type . '.*.yml';
      $files = glob($pattern);

      foreach ($files as $file) {
        $config = Yaml::parseFile($file);

        if (!empty($config['content'])) {
          foreach ($config['content'] as $field_name => $field_config) {
            if (!empty($field_config['settings']['placeholder'])) {
              $strings[] = [
                'string' => $field_config['settings']['placeholder'],
                'source' => basename($file) . ':' . $field_name,
                'type' => 'field_placeholder',
                'entity_type' => $entity_type,
              ];
            }
          }
        }
      }
    }

    return $strings;
  }

  /**
   * Scans field_group configuration files for group labels.
   *
   * @return array
   *   Array of found strings with metadata.
   */
  protected function scanFieldGroupConfigs(): array {
    $strings = [];

    // Get config path from settings.
    // Note: Settings::get('config_sync_directory') returns an absolute path.
    $config_path = Settings::get('config_sync_directory');

    if (!is_dir($config_path)) {
      return $strings;
    }

    // Scan field_group configurations for node types.
    // Pattern: core.entity_form_display.node.*.yml and core.entity_view_display.node.*.yml.
    $patterns = [
      $config_path . '/core.entity_form_display.node.*.yml',
      $config_path . '/core.entity_view_display.node.*.yml',
    ];

    foreach ($patterns as $pattern) {
      $files = glob($pattern);

      foreach ($files as $file) {
        $config = Yaml::parseFile($file);

        // Check for field_group in third_party_settings.
        if (!empty($config['third_party_settings']['field_group'])) {
          foreach ($config['third_party_settings']['field_group'] as $group_name => $group_config) {
            if (!empty($group_config['label'])) {
              $strings[] = [
                'string' => $group_config['label'],
                'source' => basename($file) . ':field_group:' . $group_name,
                'type' => 'field_group_label',
              ];
            }
          }
        }
      }
    }

    return $strings;
  }

  /**
   * Registers all found strings with Drupal's translation system.
   *
   * @param array $strings
   *   Array of strings to register.
   *
   * @return int
   *   Number of strings registered.
   */
  public function registerStrings(array $strings): int {
    $registered = 0;

    foreach ($strings as $string_data) {
      if (empty($string_data['string'])) {
        continue;
      }

      // Create or update the source string in the locale storage.
      // Use empty context to match frontend usage of t() and |t filter.
      $source = $this->localeStorage->findString(['source' => $string_data['string'], 'context' => '']);

      if (!$source) {
        // Create new source string with empty context.
        $source = $this->localeStorage->createString([
          'source' => $string_data['string'],
          'context' => '',
        ]);
        $source->save();
        $registered++;
      }
    }

    return $registered;
  }

  /**
   * Clears the translation string cache.
   */
  public function clearCache(): void {
    $this->cache->delete('transparse:strings');
    // Cache tags are automatically invalidated when cache items are deleted.
  }

}
