<?php

namespace Drupal\Tests\transparse\Unit;

use Drupal\Tests\UnitTestCase;
use Drupal\transparse\Service\TranslationParser;
use org\bovigo\vfs\vfsStream;

/**
 * Tests the Twig file scanning functionality.
 *
 * @group transparse
 */
class TwigFileScannerTest extends UnitTestCase {

  /**
   * Virtual file system directory.
   *
   * @var \org\bovigo\vfs\vfsStreamDirectory
   */
  protected $vfsRoot;

  /**
   * The translation parser service.
   *
   * @var \Drupal\transparse\Service\TranslationParser
   */
  protected $translationParser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Set up virtual file system for testing.
    $this->vfsRoot = vfsStream::setup('theme');

    // Mock dependencies.
    $file_system = $this->createMock('\Drupal\Core\File\FileSystemInterface');
    $config_factory = $this->createMock('\Drupal\Core\Config\ConfigFactoryInterface');
    $theme_handler = $this->createMock('\Drupal\Core\Extension\ThemeHandlerInterface');
    $language_manager = $this->createMock('\Drupal\Core\Language\LanguageManagerInterface');
    $string_translation = $this->createMock('\Drupal\Core\StringTranslation\TranslationInterface');
    $cache = $this->createMock('\Drupal\Core\Cache\CacheBackendInterface');
    $locale_storage = $this->createMock('\Drupal\locale\StringStorageInterface');
    $logger = $this->createMock('\Drupal\Core\Logger\LoggerChannelFactoryInterface');

    // Create the translation parser service.
    $this->translationParser = new TranslationParser(
      $file_system,
      $config_factory,
      $theme_handler,
      $language_manager,
      $string_translation,
      $cache,
      $locale_storage,
      $logger
    );
  }

  /**
   * Tests scanning Twig files for translatable strings.
   */
  public function testScanTwigFiles() {
    // Create a mock Twig file with various translatable patterns.
    $twigContent = '
      <h1>{{ "Welcome to our site"|t }}</h1>
      <p>{% trans %}This is a translatable paragraph.{% endtrans %}</p>
      <img src="image.jpg" alt={{"Product image"|t}} />
      <a href="#" title="{{ "Click here"|t }}">Link</a>
    ';

    vfsStream::newFile('template.twig')
      ->at($this->vfsRoot)
      ->setContent($twigContent);

    // Use Reflection to access the protected method.
    $reflection = new \ReflectionClass($this->translationParser);
    $method = $reflection->getMethod('scanTwigFiles');
    $method->setAccessible(TRUE);

    // Invoke the method with the virtual directory path.
    $strings = $method->invoke($this->translationParser, $this->vfsRoot->url());

    // Assertions.
    $this->assertCount(4, $strings);

    // Check for specific strings.
    $found_strings = array_column($strings, 'string');
    $this->assertContains('Welcome to our site', $found_strings);
    $this->assertContains('This is a translatable paragraph.', $found_strings);
    $this->assertContains('Product image', $found_strings);
    $this->assertContains('Click here', $found_strings);
  }

}
