<?php

namespace Drupal\Tests\transparse\Unit;

use Drupal\Tests\UnitTestCase;
use Drupal\transparse\Service\TranslationParser;
use org\bovigo\vfs\vfsStream;

/**
 * Tests the TranslationParser service.
 *
 * @group transparse
 */
class TranslationParserTest extends UnitTestCase {

  /**
   * The translation parser service.
   *
   * @var \Drupal\transparse\Service\TranslationParser
   */
  protected $translationParser;

  /**
   * Mock locale storage.
   *
   * @var \Drupal\locale\StringStorageInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $localeStorage;

  /**
   * Mock cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $cache;

  /**
   * Virtual file system directory.
   *
   * @var \org\bovigo\vfs\vfsStreamDirectory
   */
  protected $vfsRoot;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Mock dependencies.
    $file_system = $this->createMock('\Drupal\Core\File\FileSystemInterface');
    $config_factory = $this->createMock('\Drupal\Core\Config\ConfigFactoryInterface');
    $theme_handler = $this->createMock('\Drupal\Core\Extension\ThemeHandlerInterface');
    $language_manager = $this->createMock('\Drupal\Core\Language\LanguageManagerInterface');
    $string_translation = $this->createMock('\Drupal\Core\StringTranslation\TranslationInterface');
    $this->cache = $this->createMock('\Drupal\Core\Cache\CacheBackendInterface');
    $this->localeStorage = $this->createMock('\Drupal\locale\StringStorageInterface');
    $logger = $this->createMock('\Drupal\Core\Logger\LoggerChannelFactoryInterface');

    // Set up mock logger channel.
    $logger_channel = $this->createMock('\Drupal\Core\Logger\LoggerChannelInterface');
    $logger->method('get')->willReturn($logger_channel);

    // Mock default theme config.
    $config = $this->createMock('\Drupal\Core\Config\Config');
    $config->method('get')->with('default')->willReturn('stark');
    $config_factory->method('get')->with('system.theme')->willReturn($config);

    // Create the translation parser service.
    $this->translationParser = new TranslationParser(
      $file_system,
      $config_factory,
      $theme_handler,
      $language_manager,
      $string_translation,
      $this->cache,
      $this->localeStorage,
      $logger
    );

    $this->vfsRoot = vfsStream::setup('theme');
  }

  /**
   * Tests the scanTheme method.
   */
  public function testScanTheme() {
    // Create a dummy twig file.
    vfsStream::newFile('test.twig')
      ->at($this->vfsRoot)
      ->setContent('{{ "Hello World"|t }}');

    // Call scanTheme with the vfs path.
    // Note: scanTheme calls other scan methods which use Settings::get.
    // In a unit test, Settings::get returns NULL, so those methods
    // return empty arrays. This effectively isolates the Twig scanning
    // part which uses the passed path.
    $strings = $this->translationParser->scanTheme($this->vfsRoot->url());

    $this->assertIsArray($strings);
    $this->assertNotEmpty($strings);
    $this->assertEquals('Hello World', $strings[0]['string']);
  }

  /**
   * Tests the registerStrings method with sample data.
   */
  public function testRegisterStrings() {
    $strings = [
      [
        'string' => 'Test String',
        'source' => 'test.twig',
        'type' => 'twig_filter',
      ],
    ];

    // Mock findString to return null (string doesn't exist).
    $this->localeStorage->expects($this->any())
      ->method('findString')
      ->willReturn(NULL);

    // Mock string object.
    $source_string = $this->createMock('\Drupal\locale\SourceString');
    $source_string->expects($this->once())->method('save');

    // Mock createString to return our mock string object.
    $this->localeStorage->expects($this->once())
      ->method('createString')
      ->with([
        'source' => 'Test String',
        'context' => '',
      ])
      ->willReturn($source_string);

    $count = $this->translationParser->registerStrings($strings);
    $this->assertEquals(1, $count);
  }

  /**
   * Tests the clearCache method.
   */
  public function testClearCache() {
    $this->cache->expects($this->once())
      ->method('delete')
      ->with('transparse:strings');

    $this->translationParser->clearCache();
  }

}
