<?php

namespace Drupal\trash;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManager;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\Sql\SqlEntityStorageInterface;
use Drupal\Core\PhpStorage\PhpStorageFactory;

/**
 * Provides custom storage handlers for trash-supported entity types.
 *
 * Needs to `extend EntityTypeManager` temporarily until
 * https://www.drupal.org/project/drupal/issues/3564264 is fixed.
 * Make sure there are no parent::method() calls in the meantime!
 */
final class TrashEntityTypeManager extends EntityTypeManager implements EntityTypeManagerInterface {

  /**
   * {@inheritdoc}
   *
   * @phpstan-ignore-next-line pluginManagerSetsCacheBackend.missingCacheBackend
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $inner,
  ) {
  }

  /**
   * Contains instantiated storage handlers keyed by entity type.
   *
   * @var array
   */
  protected $storageHandlers = [];

  /**
   * {@inheritdoc}
   */
  public function clearCachedDefinitions() {
    $this->inner->clearCachedDefinitions();
    $this->clearTrashStorage();
  }

  /**
   * {@inheritdoc}
   */
  public function useCaches($use_caches = FALSE) {
    $this->inner->useCaches($use_caches);
    if (!$use_caches) {
      $this->clearTrashStorage();
    }
  }

  /**
   * Clears the storage handlers generated by Trash.
   */
  protected function clearTrashStorage(): void {
    // @phpstan-ignore-next-line
    $state = \Drupal::state();
    if ($state->get('trash.class_suffix')) {
      $state->delete('trash.class_suffix');
      $trash_php_storage = PhpStorageFactory::get('trash');
      $trash_php_storage->deleteAll();
    }
    $this->storageHandlers = [];
  }

  /**
   * {@inheritdoc}
   */
  public function getStorage($entity_type_id) {
    return $this->getHandler($entity_type_id, 'storage');
  }

  /**
   * {@inheritdoc}
   */
  public function getHandler($entity_type_id, $handler_type) {
    if ($handler_type !== 'storage') {
      return $this->inner->getHandler($entity_type_id, $handler_type);
    }

    if (!isset($this->storageHandlers[$entity_type_id])) {
      $definition = $this->getDefinition($entity_type_id);
      $class = $definition->getHandlerClass($handler_type);
      if (!$class) {
        throw new InvalidPluginDefinitionException($entity_type_id, sprintf('The "%s" entity type did not specify a %s handler.', $entity_type_id, $handler_type));
      }
      $this->storageHandlers[$entity_type_id] = $this->createHandlerInstance($class, $definition);
    }

    return $this->storageHandlers[$entity_type_id];
  }

  /**
   * {@inheritdoc}
   */
  public function createHandlerInstance($class, ?EntityTypeInterface $definition = NULL) {
    // @phpstan-ignore-next-line
    if (\Drupal::service('trash.manager')->isEntityTypeEnabled($definition)
      && is_subclass_of($class, SqlEntityStorageInterface::class)) {
      $class = _trash_generate_storage_class($class);
    }

    return $this->inner->createHandlerInstance($class, $definition);
  }

  /**
   * {@inheritdoc}
   */
  public function hasDefinition($plugin_id) {
    return $this->inner->hasDefinition($plugin_id);
  }

  /**
   * {@inheritdoc}
   */
  public function getAccessControlHandler($entity_type_id) {
    return $this->inner->getAccessControlHandler($entity_type_id);
  }

  /**
   * {@inheritdoc}
   */
  public function getViewBuilder($entity_type_id) {
    return $this->inner->getViewBuilder($entity_type_id);
  }

  /**
   * {@inheritdoc}
   */
  public function getListBuilder($entity_type_id) {
    return $this->inner->getListBuilder($entity_type_id);
  }

  /**
   * {@inheritdoc}
   */
  public function getFormObject($entity_type_id, $operation) {
    return $this->inner->getFormObject($entity_type_id, $operation);
  }

  /**
   * {@inheritdoc}
   */
  public function getRouteProviders($entity_type_id) {
    return $this->inner->getRouteProviders($entity_type_id);
  }

  /**
   * {@inheritdoc}
   */
  public function hasHandler($entity_type_id, $handler_type) {
    return $this->inner->hasHandler($entity_type_id, $handler_type);
  }

  /**
   * {@inheritdoc}
   */
  public function getDefinition($entity_type_id, $exception_on_invalid = TRUE) {
    return $this->inner->getDefinition($entity_type_id, $exception_on_invalid);
  }

  /**
   * {@inheritdoc}
   */
  public function getDefinitions() {
    return $this->inner->getDefinitions();
  }

  /**
   * {@inheritdoc}
   */
  public function createInstance($plugin_id, array $configuration = []) {
    return $this->inner->createInstance($plugin_id, $configuration);
  }

  /**
   * {@inheritdoc}
   */
  public function getInstance(array $options) {
    return $this->inner->getInstance($options);
  }

  /**
   * Default EntityTypeManager has some naughty methods not on interface.
   */
  public function getActiveDefinition($entity_type_id) {
    // @phpstan-ignore-next-line method.notFound
    return $this->inner->getActiveDefinition($entity_type_id);
  }

}
