<?php

declare(strict_types=1);

namespace Drupal\trash;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\DependencyInjection\ServiceProviderBase;
use Drupal\trash\EventSubscriber\TrashIgnoreSubscriber;
use Drupal\trash\Handler\TrashHandlerPass;
use Drupal\trash\LayoutBuilder\TrashInlineBlockUsage;
use Symfony\Component\DependencyInjection\Reference;

/**
 * Alters container services.
 */
class TrashServiceProvider extends ServiceProviderBase {

  /**
   * {@inheritdoc}
   */
  public function register(ContainerBuilder $container) {
    $container->addCompilerPass(new TrashHandlerPass());
  }

  /**
   * {@inheritdoc}
   */
  public function alter(ContainerBuilder $container): void {
    if ($container->hasDefinition('workspaces.information')) {
      $container->register('trash.workspaces.information', TrashWorkspaceInformation::class)
        ->setPublic(FALSE)
        ->setDecoratedService('workspaces.information', NULL, 100)
        ->addArgument(new Reference('trash.workspaces.information.inner'))
        ->addArgument(new Reference('trash.manager'));
    }

    if ($container->hasDefinition('workspaces.manager')) {
      $container->register('trash.workspaces.manager', TrashWorkspaceManager::class)
        ->setPublic(FALSE)
        ->setDecoratedService('workspaces.manager', NULL, 100)
        ->addArgument(new Reference('trash.workspaces.manager.inner'))
        ->addArgument(new Reference('trash.manager'));
    }

    if ($container->hasDefinition('inline_block.usage')) {
      $container->register('trash.inline_block.usage', TrashInlineBlockUsage::class)
        ->setPublic(FALSE)
        ->setDecoratedService('inline_block.usage')
        ->addArgument(new Reference('trash.inline_block.usage.inner'))
        ->addArgument(new Reference('trash.manager'));
    }

    if ($container->hasDefinition('wse_menu.tree_storage')) {
      $container->getDefinition('wse_menu.tree_storage')
        ->setClass(TrashWseMenuTreeStorage::class)
        ->addMethodCall('setTrashManager', [new Reference('trash.manager')]);
    }

    // Ensure the trash ignore subscriber is one of the first definitions used
    // after authentication. This is necessary because in the event that two
    // subscriber event listeners have the same priority, then the one which was
    // registered first takes precedence. We must ensure that the ignore
    // subscriber is one of the first subscribers with a priority of 299.
    $trash_ignore_subscriber = TrashIgnoreSubscriber::class;

    // This is one of the earliest 'kernel.request' event listeners with a
    // priority of 299, so we must ensure the ignore subscriber is added
    // before it.
    $target = 'system.timezone_resolver';

    if ($container->hasDefinition($trash_ignore_subscriber) && $container->hasDefinition($target)) {
      $definitions = $container->getDefinitions();
      // Move the Trash ignore subscriber before 'system.timezone_resolver', so
      // it runs first among priority-299 'kernel.request' listeners.
      $trash_definition = [$trash_ignore_subscriber => $definitions[$trash_ignore_subscriber]];
      unset($definitions[$trash_ignore_subscriber]);
      $pos = array_search($target, array_keys($definitions), TRUE);
      assert(is_int($pos));
      $container->setDefinitions(
        array_slice($definitions, 0, $pos, TRUE) +
        $trash_definition +
        array_slice($definitions, $pos, NULL, TRUE)
      );
    }
  }

}
