<?php

declare(strict_types = 1);

namespace Drupal\trash_workflows\Hook;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;

/**
 * Hook implementation for extending the Workflow Buttons widget. 
 */
class AlterWorkflowButtonsWidgetForm {

  /**
   * Implements hook_field_widget_single_element_WIDGET_TYPE_form_alter() for the workflow_buttons widget.
   *
   * The hook_field_widget_single_element_form_alter hook targeted to the Workflow Buttons widget form.
   */
  #[Hook('field_widget_single_element_workflow_buttons_form_alter')]
  public function fieldWidgetWorkflowButtonsFormAlter(array &$element, FormStateInterface $form_state, array $context) {
    // @TODO get_class($this) or get_called_class() should work here more
    // elegantly, but i'd like to figure out which is indeed best and do that
    // everywhere.  Both are widely used in contrib, by the same modules even,
    // so there must be a difference?  (Core mostly only in tests.)
    $element['#process'][] = [self::class, "processActions"];
  }

  /**
   * Process callback on the workflow buttons widget element to modify actions.
   *
   * This *may* all just be carryover from before we provided our own widget,
   * but Workflow Buttons does most of its work in a process callback so any
   * changes or additions to that work must do the same.
   */
  public static function processActions($element, FormStateInterface $form_state, array &$form) {
    if (isset($form['actions']['delete'])) {
      // To avoid confusing admins who can see both delete buttons, we update
      // the delete link to be clear that it is the real, permanent delete.
      // We want it to do this even (especially) when content is in the trash
      // already, and there is no harm in being more explicit even for content
      // types that may not be using the trash workflow.
      $form['actions']['delete']['#title'] = t('Permanently delete');
    }

    // If there is no trash moderation state action, nothing for us to do.
    // @TODO this is yet another hard-coded transition name, which should be
    // made configurable, probably would have to be a separate form where you
    // can select from existing transitions which should be trash.
    if (!isset($form['actions']['moderation_state_trash'])) {
      // It is *very* important we return $element here; the $form is passed by
      // reference but we need to give $element back even if we don't touch it.
      return $element;
    }
    $button =& $form['actions']['moderation_state_trash'];
    $button['#type'] = 'image_button';
    unset($button['#value']);
    $button['#button_type'] = 'danger';
    // This styles the delete button much like the default delete link, at
    // least in Claro.
    // See https://www.drupal.org/project/workflow_buttons/issues/3092099
    $button['#attributes'] = [
      'class' => [
        'button',
        'submit-trash',
      ],
      'title' => t('Trash: Deleted content can be restored from the trash later'),
      'alt' => t('Trash'),
    ];
    $path = \Drupal::service('extension.path.resolver')->getPath('module', 'trash_workflows');
    $button['#src'] = $path . '/img/trash.svg';

    $button['#attached']['library'][] = 'trash_workflows/delete-button';

    return $element;
  }

}