<?php

declare(strict_types=1);

namespace Drupal\trinion_bpmn_import\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\node\Entity\Node;
use Drupal\taxonomy\Entity\Term;

/**
 * BPMN import class
 */
final class BpmnImportController extends ControllerBase {

  protected $namespaces;
  protected Node $node;
  protected array $roleElements = [];
  protected array $elementsIds = [];
  protected array $bpElementsIds = [];

  public function __construct(Node $node) {
    $this->node = $node;
  }

  public function import() {
    $file_name = $this->node->get('field_tbi_fayl_importa')->first()->entity->getFileUri();
    /** @var \SimpleXMLElement $xml */
    $file = file_get_contents($file_name);
    $xml = simplexml_load_string($file);
    $this->namespaces = $xml->getNamespaces(true);
    $namespaces = $xml->getNamespaces(true);

//    foreach ($xml->children($namespaces["bpmn"]) as $root_element_name => $root_line) {
//      dump($root_element_name);
//      foreach ($root_line->children($namespaces["bpmn"]) as $m => $l) {
//        dump($m);
//      }
//    }
//    exit;

    $query = \Drupal::entityQuery('node')
      ->condition('field_tbi_shablon_biznes_process', $this->node->id());
    $old_nodes = $query->accessCheck()->execute();
    $nodes_exists = [];

    foreach ($xml->children($namespaces["bpmn"]) as $root_element_name => $root_line) {
      if ($root_element_name == 'process') {
        $biznes_process_id = (string)$root_line->attributes()['id'];
        $this->bpElementsIds[$biznes_process_id] = [
          'participant_id' => NULL,
          'inner' => NULL,
          'outer' => NULL,
        ];
        if ((bool)$root_line->children($namespaces["bpmn"])->laneSet) {
          foreach ($root_line->children($namespaces["bpmn"])->laneSet as $line)
            $this->elementLaneSet($line);
        } else {
          $this->elementLaneSet(FALSE, $this->node->label());
        }

        foreach ($root_line->children($namespaces["bpmn"]) as $element_name => $line) {
          if (in_array($element_name, ['laneSet', 'sequenceFlow']))
            continue;
          $name = 'element' . ucwords($element_name);
          if (preg_match('/^.*Event$/', $element_name)) {
            $nodes_exists[] = $this->wrapperElementEvent($element_name, $line, $biznes_process_id);
          }
          elseif (preg_match('/^.*T|task|callActivity|subProcess$/', $element_name))
            $nodes_exists[] = $this->wrapperElementTask($element_name, $line);
          elseif (preg_match('/^.*Gateway$/', $element_name))
            $nodes_exists[] = $this->wrapperElementGateway($element_name, $line);
          elseif (method_exists($this, $name)) {
            $nodes_exists[] = $this->$name($line);
          }
        }
        foreach ($root_line->children($namespaces["bpmn"])->sequenceFlow as $line)
          $nodes_exists[] = $this->elementSequenceFlow($line);
      }
    }

    foreach ($xml->children($namespaces["bpmn"]) as $root_element_name => $root_line) {
      if ($root_element_name == 'collaboration') {
        foreach ($root_line->children($namespaces["bpmn"])->participant as $line) {
          $biznes_process_attrs = $line->attributes();
          $biznes_process_id = (string)$biznes_process_attrs['processRef'];
          $this->bpElementsIds[$biznes_process_id]['participant_id'] = (string)$biznes_process_attrs['id'];
        }
      }
    }

    foreach ($xml->children($namespaces["bpmn"]) as $root_element_name => $root_line) {
      if ($root_element_name == 'collaboration') {
        foreach ($root_line->children($namespaces["bpmn"])->messageFlow as $line) {
          $message_flows = $this->elementMessageFlow($line);
          $nodes_exists = array_merge($nodes_exists, $message_flows);
        }
      }
    }

    foreach (array_diff($old_nodes, $nodes_exists) as $nid)
      Node::load($nid)->delete();
  }

  private function elementLaneSet($line, $role_name = NULL) {
    /** @var \SimpleXMLElement $lane */
    $query = \Drupal::entityQuery('node')
      ->condition('vid', 'biznes_process_rol')
      ->condition('field_tbi_shablon_biznes_process', $this->node->id());
    $old_roles = $query->accessCheck()->execute();
    $role_exists = [];
    if ($line) {
      foreach ($line->children($this->namespaces["bpmn"]) as $lane) {
        $attrs = $lane->attributes();
        $role_name = $this->node->label() . ' / ' . (string)$attrs['name'];
        $role_tid = $this->saveRole($role_name);
        $role_exists[] = $role_tid;
        foreach ($lane as $lane_element) {
          $this->roleElements[(string)$lane_element] = $role_tid;
        }
      }
    }
    else {
      $role_tid = $this->saveRole($role_name);
      $role_exists[] = $role_tid;
      $this->roleElements[0] = $role_tid;
    }
    foreach ($old_roles as $old_role) {
      if (!in_array($old_role, $role_exists))
        Term::load($old_role)->delete();
    }
  }

  private function wrapperElementEvent($element_name, $line, $biznes_process_id) {
    $attrs = $line->attributes();
    $element_id = (string)$attrs['id'];

    if ($element_name == 'startEvent')
      $this->bpElementsIds[$biznes_process_id]['inner'] = $element_id;
    if ($element_name == 'endEvent')
      $this->bpElementsIds[$biznes_process_id]['outer'] = $element_id;

    $event_name = (string)$attrs['name'];
    foreach ($line->children($this->namespaces["bpmn"]) as $name => $val) {
      if (preg_match('/.+EventDefinition/', $name))
        $event_type = $name;
    }

    $fields = [
      'field_tbi_biznes_process_rol' => isset($this->roleElements[$element_id]) ? $this->roleElements[$element_id] : reset($this->roleElements),
      'field_tbi_shablon_biznes_process' => $this->node->id(),
      'field_tbi_mesto_sobytiya' => $element_name,
    ];
    if (isset($event_type))
      $fields['field_tbi_tip_sobytiya'] = $event_type;

    if ($event_name == '')
      $event_name = $element_name;

    $element_id = (string)$attrs['id'];
    $id = $this->saveNodeElement('trinion_bpmn_sobytie', $element_id, $event_name, $this->node->id(), $fields);
    $this->elementsIds[$element_id] = $id;
    return $id;
  }

  private function wrapperElementTask($element_name, $line) {
    $attrs = $line->attributes();
    $event_name = (string)$attrs['name'];

    $fields = [
      'field_tbi_biznes_process_rol' => isset($this->roleElements[(string)$attrs['id']]) ? $this->roleElements[(string)$attrs['id']] : reset($this->roleElements),
      'field_tbi_shablon_biznes_process' => $this->node->id(),
      'field_tbi_tip_zadachi' => $element_name,
    ];
    if ($event_name == '')
      $event_name = $element_name;

    $element_id = (string)$attrs['id'];
    $id = $this->saveNodeElement('bpmn_zadacha', $element_id, $event_name, $this->node->id(), $fields);
    $this->elementsIds[$element_id] = $id;
    return $id;
  }

  private function wrapperElementGateway($element_name, $line) {
    $attrs = $line->attributes();
    $event_name = (string)$attrs['name'];

    $fields = [
      'field_tbi_biznes_process_rol' => isset($this->roleElements[(string)$attrs['id']]) ? $this->roleElements[(string)$attrs['id']] : reset($this->roleElements),
      'field_tbi_shablon_biznes_process' => $this->node->id(),
      'field_tbi_tip_shlyuza' => $element_name,
    ];

    if ($event_name == '')
      $event_name = $element_name;

    $element_id = (string)$attrs['id'];
    $id = $this->saveNodeElement('bpmn_shlyuz', $element_id, $event_name, $this->node->id(), $fields);
    $this->elementsIds[$element_id] = $id;
    return $id;
  }

  private function elementSequenceFlow($line) {
    $attrs = $line->attributes();
    $event_name = (string)$attrs['name'];
    $fields = [
      'field_tbi_shablon_biznes_process' => $this->node->id(),
      'field_tbi_vhodyashchaya_ssylka' => $this->elementsIds[(string)$attrs['sourceRef']],
      'field_tbi_ishodyashchaya_ssylka' => $this->elementsIds[(string)$attrs['targetRef']],
    ];

    if ($event_name == '')
      $event_name = 'sequenceFlow';

    return $this->saveNodeElement('bpmn_potok', (string)$attrs['id'], $event_name, $this->node->id(), $fields);
  }

  private function elementMessageFlow($line) {
    $attrs = $line->attributes();
    $source_ref = (string)$attrs['sourceRef'];
    $target_ref = (string)$attrs['targetRef'];
    if (isset($this->elementsIds[$source_ref])) {
      $source_element_id1 = $this->elementsIds[$source_ref];
      $target_element_id2 = $source_element_id1;
    }
    else {
      foreach ($this->bpElementsIds as $bp_id => $bp_data) {
        if ($bp_data['participant_id'] == $source_ref) {
          $source_element_id1 = $this->elementsIds[$bp_data['outer']];
        }
      }
    }
    foreach ($this->bpElementsIds as $bp_id => $bp_data) {
      if ($bp_data['participant_id'] == $source_ref) {
        $target_element_id2 = $this->elementsIds[$bp_data['inner']];
      }
    }

    if (isset($this->elementsIds[$target_ref])) {
      $target_element_id1 = $this->elementsIds[$source_ref];
    }
    else {
      foreach ($this->bpElementsIds as $bp_id => $bp_data) {
        if ($bp_data['participant_id'] == $target_ref) {
          $target_element_id1 = $this->elementsIds[$bp_data['inner']];
        }
      }
    }

    foreach ($this->bpElementsIds as $bp_id => $bp_data) {
      if ($bp_data['participant_id'] == $target_ref) {
        $source_element_id2 = $this->elementsIds[$bp_data['outer']];
      }
    }
//    dump($source_element_id1, $target_element_id1);
//    dump($source_element_id2, $target_element_id2);
//    dump($this->bpElementsIds);

    $event_name = (string)$attrs['name'];
    $fields = [
      'field_tbi_shablon_biznes_process' => $this->node->id(),
      'field_tbi_vhodyashchaya_ssylka' => $source_element_id1,
      'field_tbi_ishodyashchaya_ssylka' => $target_element_id1,
    ];

    if ($event_name == '')
      $event_name = 'sequenceFlow';
    $flows = [];
    $flows[] = $this->saveNodeElement('bpmn_potok', (string)$attrs['id'], $event_name, $this->node->id(), $fields);

    $fields = [
      'field_tbi_shablon_biznes_process' => $this->node->id(),
      'field_tbi_vhodyashchaya_ssylka' => $source_element_id2,
      'field_tbi_ishodyashchaya_ssylka' => $target_element_id2,
    ];

    if ($event_name == '')
      $event_name = 'sequenceFlow';
    $flows[] = $this->saveNodeElement('bpmn_potok', (string)$attrs['id'], $event_name, $this->node->id(), $fields);

    return $flows;
  }

  private function saveRole($role_name) {
    $role_exists = \Drupal::entityTypeManager()->getStorage('taxonomy_term')->loadByProperties(['name' => $role_name, 'vid' => 'biznes_process_rol']);
    if ($role_exists) {
      $role = reset($role_exists);
      $role->name = $role_name;
      $role->save();
    }
    else {
      $role = Term::create([
        'vid' => 'biznes_process_rol',
        'name' => $role_name,
        'status' => 1,
        'field_tbi_shablon_biznes_process' => $this->node->id(),
      ]);
      $id = $role->save();
    }
    $id = $role->id();
    return $id;
  }

  private function saveNodeElement($type, $id, $name, $shablon_tid, $fields) {
    $node_exists = \Drupal::entityTypeManager()->getStorage('node')->loadByProperties(['title' => $id, 'type' => $type, 'field_tbi_shablon_biznes_process' => $shablon_tid]);
    if ($node_exists) {
      $node = reset($node_exists);
      $node->title = $id;
      $node->field_tbi_id = $name;
      foreach ($fields as $field => $value)
        $node->{$field} = $value;
      $node->save();
    }
    else {
      $node_data = [
        'type' => $type,
        'title' => $name,
        'field_tbi_id' => $id,
        'uid' => \Drupal::currentUser()->id(),
        'status' => 1,
      ];
      $node_data += $fields;

      $node = Node::create($node_data);
      $node->save();
    }
    $id = $node->id();
    return $id;
  }
}
