<?php

namespace Drupal\tripal_chado\ChadoCustomTables;

use \Drupal\tripal_chado\Database\ChadoConnection;

/**
 * Handles CRUD operations on custom tables in the Chado schema.
 */
class ChadoCustomTable {

  /**
   * The name of the table.
   *
   * @var string
   */
  private $table_name;

  /**
   * The ID of the table.
   *
   * @var int
   */
  private $table_id;

  /**
   * The name of the Chado schema to use.
   *
   * @var string
   */
  private $chado_schema;

  /**
   * Initializes the service object with a table name.
   *
   * This object will work with the custom table in the default Chado schema.
   * Be sure to call the setSchemaName() on the ChadoConnection object to
   * ensure the custom table is managed in the correct Chado instance.
   *
   * @param string $table_name
   *   The name of the custom table.
   * @param string|null $chado_schema
   *   Optional. The chado schema where the custom table will live. If no
   *   schema is specified then the default schema is used.
   */
  public function __construct($table_name, ?string $chado_schema = NULL) {
    $this->table_name = NULL;
    $this->table_id = NULL;
    $this->chado_schema = NULL;

    if (!$table_name) {
      throw new \Exception('Please provide a value for the $table_name argument');
    }

    $this->table_name = $table_name;
    $this->chado_schema = $chado_schema;
    if (!$chado_schema) {
      $chado = \Drupal::service('tripal_chado.database');
      $this->chado_schema = $chado->getSchemaName();
    }
    $this->setTableId();

    // If this table doesn't exist (i.e. it has no ID) then create
    // an empty record for it.
    if (!$this->table_id) {
      $public = \Drupal::database();
      $insert = $public->insert('tripal_custom_tables');
      $insert->fields([
        'table_name' => $this->table_name,
        'schema' => '',
        'chado' => $this->chado_schema,
      ]);
      $table_id = $insert->execute();
      if (!$table_id) {
        throw new \Exception('Could not add the custom table, "' . $this->table_name .
            '" for the Chado schema "' . $this->chado_schema . '".');
      }
      $this->setTableId();
    }
  }

  /**
   * Returns a ChadoConnection object with the correct schema set.
   *
   * This is just a helper function for this class to make sure the
   * Chado schema is set as requested anytime the object is needed.
   *
   * @return \Drupal\tripal_chado\Database\ChadoConnection
   *   A Chado database connection.
   */
  protected function getChado(): ChadoConnection {
    $chado = \Drupal::service('tripal_chado.database');
    if ($this->chado_schema) {
      $chado->setSchemaName($this->chado_schema);
    }
    return $chado;
  }

  /**
   * Sets the private table_id member variable.
   */
  private function setTableId() {
    $public = \Drupal::database();
    $query = $public->select('tripal_custom_tables', 'ct');
    $query->fields('ct', ['table_id']);
    $query->condition('ct.table_name', $this->table_name);
    $query->condition('ct.chado', $this->chado_schema);
    $results = $query->execute();
    $this->table_id = $results->fetchField();

  }

  /**
   * Retrieves the numeric ID of the custom table.
   *
   * @return int
   *   The table's numeric ID.
   */
  public function getTableId(): int {
    return (int) $this->table_id;
  }

  /**
   * Retrieves the name of the custom table.
   *
   * @return string
   *   The name of the custom table.
   */
  public function getTableName(): string {
    return $this->table_name;
  }

  /**
   * Retrieves the name of the Chado schema in which this table lives.
   *
   * @return string
   *   The chado schema name.
   */
  public function getChadoSchema(): string {
    return $this->chado_schema;
  }

  /**
   * Toggles the custom table's locked (formerly hidden) status.
   *
   * Tables that are locked are meant to be managed internally by the
   * Tripal module that created it and should not be changed or deleted by
   * the end-user.
   *
   * @param bool $lock
   *   Set to TRUE to lock the table. Set to FALSE to allow the table to be
   *   edited by the end-user.
   */
  public function setLocked($lock = FALSE) {
    $public = \Drupal::database();
    $update = $public->update('tripal_custom_tables');
    $update->fields(['locked' => $lock == TRUE ? 1 : 0]);
    $update->condition('table_name', $this->table_name);
    $update->condition('chado', $this->chado_schema);
    $update->execute();
  }

  /**
   * Indicates if the custom table is locked (formerly hidden).
   *
   * Tables that are locked are meant to be managed internally by the
   * Tripal module that created it and should not be changed or deleted by
   * the end-user.
   *
   * @return bool
   *   The locked status, TRUE if locked, FALSE if not.
   */
  public function isLocked(): bool {
    $public = \Drupal::database();
    $query = $public->select('tripal_custom_tables', 'tct');
    $query->fields('tct', ['locked']);
    $query->condition('tct.table_name', $this->table_name);
    $query->condition('tct.chado', $this->chado_schema);
    $locked = $query->execute()->fetchField();

    if ($locked == 1) {
      return TRUE;
    }
    return FALSE;
  }

  /**
   * Retrieves the schema for the custom table.
   *
   * If return value is empty then it means the table schema has not yet
   * been provided or the init() function has not been called. Use
   * the setTableSchema() function to provide one.
   *
   * @return array
   *   The schema array.
   */
  public function getTableSchema(): array {
    $logger = \Drupal::service('tripal.logger');
    if (!$this->table_id) {
      $logger->error('Cannot get the the custom table schema. Please, first run the init() function.');
      return [];
    }
    $public = \Drupal::database();
    $query = $public->select('tripal_custom_tables', 'tct');
    $query->fields('tct', ['schema']);
    $query->condition('tct.table_name', $this->table_name);
    $query->condition('tct.chado', $this->chado_schema);
    $table_schema = $query->execute()->fetchField();
    if (!$table_schema) {
      return [];
    }
    // There was a hard-to-reproduce bug with unserializing which
    // was fixed with the allowed_classes option, which provides
    // more security, and there are no classes serialized here.
    return unserialize($table_schema, ['allowed_classes' => FALSE]);
  }

  /**
   * Sets the table schema.
   *
   * When setting the table schema, the table will be created in the Chado
   * schema if it doesn't exist. If the table does exist then the $force
   * argument must be set to TRUE and the table will be dropped and recreated.
   * If not set to TRUE then no change is made to the schema or the custom
   * table. The force argument is to prevent accidental deletion and recreation
   * of tables that may have data.
   *
   * If a mistake was made in the schema definition and it needs correction
   * make sure the $force argument is set to FALSE. But be careful. If the
   * schema does not properly match the table problems may occur when using
   * the table later.
   *
   * @param array $table_schema
   *   The Drupal table schema array defining the table.
   * @param bool $force
   *   TRUE if the custom table should be dropped and recreated if it already
   *   exists.
   *
   * @return bool
   *   TRUE on successful.
   */
  public function setTableSchema(array $table_schema, bool $force = FALSE): bool {

    $logger = \Drupal::service('tripal.logger');
    if (!$this->table_id) {
      $logger->error('Cannot set the schema for the custom table. Please, first run the init() function.');
      return FALSE;
    }

    $public = \Drupal::database();
    $chado = $this->getChado();
    $transaction_chado = $chado->startTransaction();
    try {

      // Don't set the schema if it's not valid.
      $errors = ChadoCustomTable::validateTableSchema($table_schema);
      if (!empty($errors)) {
        return FALSE;
      }

      // If the table name is the same and the user isn't forcing any changes
      // then create the table if it doesn't exist. If it does exist then leave
      // the table as is and the function will later update the saved schema.
      if ($force == FALSE and $this->table_name == $table_schema['table']) {
        $table_exists = $chado->schema()->tableExists($this->table_name);
        if (!$table_exists) {
          $chado->schema()->createTable($this->table_name, $table_schema);
        }
      }

      // If the table name is the same and the user is forcing a change then
      // create the table if it doesn't exist. If it does exist then drop it
      // and recreate it.
      if ($force == TRUE and $this->table_name == $table_schema['table']) {
        if ($chado->schema()->tableExists($this->table_name)) {
          $chado->schema()->dropTable($this->table_name);
        }
        $chado->schema()->createTable($this->table_name, $table_schema);
      }

      // If the table name is different in the provided schema but the user is
      // not forcing a change then this shouldn't be allowed. We don't want to
      // update the saved schema with a table name mismatch.
      if ($force == FALSE and $this->table_name != $table_schema['table']) {
        $logger->error('Cannot change the name of the table in the schema without forcing it..');
        return FALSE;
      }

      // If the table name is different and the force argument is TRUE, then the
      // user is requesting a rename of the table. Make sure the name isn't
      // already taken. If not, then drop the old table and create the new one.
      if ($force == TRUE and $this->table_name != $table_schema['table']) {

        // First check if the new table exists and if so return FALSE.
        if ($chado->schema()->tableExists($table_schema['table'])) {
          $logger->error('Cannot rename the table as another table exists with the same name.');
          return FALSE;
        }

        // Second, if the original table exists then delete it.
        if ($chado->schema()->tableExists($this->table_name)) {
          $chado->schema()->dropTable($this->table_name);
        }
        $this->table_name = $table_schema['table'];
        $chado->schema()->createTable($this->table_name, $table_schema);
      }

      $update = $public->update('tripal_custom_tables');
      $update->fields([
        'table_name' => $table_schema['table'],
        'schema' => serialize($table_schema),
      ]);
      $update->condition('table_id', $this->table_id);
      $update->condition('chado', $chado->getSchemaName());
      $update->execute();
    }
    catch (Exception $e) {
      $transaction_chado->rollback();
      $logger->error($e->getMessage());
      return FALSE;
    }
    return TRUE;
  }

  /**
   * Ensures that the table schema is correctly formatted.
   *
   * Returns a list of messages indicating if any errors are present.
   *
   * @param array $table_schema
   *   The Drupal table schema array defining the table.
   *
   * @return array
   *   A list of error message strings indicating what is wrong with the
   *   schema. If the array is empty then no errors were detected.
   */
  public static function validateTableSchema(array $table_schema): array {

    $messages = [];
    $logger = \Drupal::service('tripal.logger');
    if (!$table_schema) {
      $message = 'The custom table schema is empty.';
      $messages[] = $message;
      $logger->error($message);
      return $messages;
    }

    if (!array_key_exists('table', $table_schema)) {
      $message = "The schema array must have key named 'table'";
      $messages[] = $message;
      $logger->error($message);
      return $messages;
    }

    if (preg_match('/[ABCDEFGHIJKLMNOPQRSTUVWXYZ]/', $table_schema['table'])) {
      $message = "Postgres will automatically change the table name to lower-case. To prevent unwanted side-effects, please rename the table with all lower-case characters.";
      $messages[] = $message;
      $logger->error($message);
    }

    // Check index length.
    if (array_key_exists('indexes', $table_schema)) {
      foreach (array_keys($table_schema['indexes']) as $index_name) {
        if (strlen($table_schema['table'] . '_' . $index_name) > 60) {
          $message = "One or more index names appear to be too long. For example: '" .
              $table_schema['table'] . '_' . $index_name . ".'  Index names are created by " .
            "concatenating the table name with the index name provided " .
            "in the 'indexes' array of the schema. Please alter any indexes that " .
            "when combined with the table name are longer than 60 characters.";
          $messages[] = $message;
          $logger->error($message);
        }
      }
    }
    return $messages;
  }

  /**
   * Destroyes the custom table completely.
   *
   * Tripal will no longer know about the table and the table will be removed
   * from Chado. After this function is executed this object is no longer
   * usable.
   *
   * @return bool
   *   TRUE if successful. FALSE otherwise.
   */
  public function delete(): bool {
    $logger = \Drupal::service('tripal.logger');
    if (!$this->table_id) {
      $logger->error('Cannot destroy the custom table. Please, first run the init() function.');
      return FALSE;
    }
    $public = \Drupal::database();
    $delete = $public->delete('tripal_custom_tables');
    $delete->condition('table_id', $this->table_id);
    $delete->execute();

    $this->deleteCustomTable();

    $this->table_id = NULL;
    $this->table_name = NULL;
    return TRUE;
  }

  /**
   * Deletes the table in Chado.
   *
   * @return bool
   *   TRUE if successful. FALSE otherwise.
   */
  private function deleteCustomTable(): bool {
    $logger = \Drupal::service('tripal.logger');
    $chado = $this->getChado();
    $transaction_chado = $chado->startTransaction();

    $table_exists = $chado->schema()->tableExists($this->table_name);
    if (!$table_exists) {
      return TRUE;
    }

    try {
      $chado->schema()->dropTable($this->table_name);
      if ($chado->schema()->tableExists($this->table_name)) {
        $logger->error('Could not delete the ' . $this->table_name . ' table. Check the database server logs.');
        return FALSE;
      }
    }
    catch (Exception $e) {
      $transaction_chado->rollback();
      $logger->error($e->getMessage());
      return FALSE;
    }
    return TRUE;
  }

}
