<?php

namespace Drupal\tripal_chado\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tripal\TripalField\Attribute\TripalFieldFormatter;
use Drupal\tripal_chado\TripalField\ChadoFormatterBase;

/**
 * Plugin implementation of default Tripal feature formatter.
 */
#[TripalFieldFormatter(
  id: 'chado_feature_formatter_default',
  label: new TranslatableMarkup('Chado feature formatter'),
  description: new TranslatableMarkup('A chado feature formatter'),
  field_types: [
    'chado_feature_type_default',
  ],
  valid_tokens: [
    '[name]',
    '[uniquename]',
    '[type]',
    '[seqlen]',
    '[md5checksum]',
    '[is_analysis]',
    '[is_obsolete]',
    '[database_name]',
    '[database_accession]',
    '[genus]',
    '[species]',
    '[infratype]',
    '[infratype_abbrev]',
    '[infraname]',
    '[abbreviation]',
    '[common_name]',
  ],
)]
class ChadoFeatureFormatterDefault extends ChadoFormatterBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    $settings = parent::defaultSettings();
    $settings['token_string'] = '[name]';
    return $settings;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    parent::viewElements($items, $langcode);
    $list = [];
    $token_string = $this->getSetting('token_string');
    $lookup_manager = \Drupal::service('tripal.tripal_entity.lookup');

    foreach ($items as $delta => $item) {
      $values = [
        'entity_id' => $item->get('entity_id')->getString(),
        'name' => $item->get('feature_name')->getString(),
        'uniquename' => $item->get('feature_uniquename')->getString(),
        'type' => $item->get('feature_type')->getString(),
        'seqlen' => $item->get('feature_seqlen')->getString(),
        'md5checksum' => $item->get('feature_md5checksum')->getString(),
        'is_analysis' => $item->get('feature_is_analysis')->getString(),
        'is_obsolete' => $item->get('feature_is_obsolete')->getString(),
        'database_name' => $item->get('feature_database_name')->getString(),
        'database_accession' => $item->get('feature_database_accession')->getString(),
        'genus' => $item->get('feature_genus')->getString(),
        'species' => $item->get('feature_species')->getString(),
        'infratype' => $item->get('feature_infraspecific_type')->getString(),
        'infraname' => $item->get('feature_infraspecific_name')->getString(),
        'abbreviation' => $item->get('feature_abbreviation')->getString(),
        'common_name' => $item->get('feature_common_name')->getString(),
        // residues is not implemented in this field since it can be millions of characters long
        // timeaccessioned, timelastmodified not implemented
      ];

      // Substitute values in token string to generate displayed string.
      $displayed_string = $token_string;
      foreach ($values as $key => $value) {
        $displayed_string = preg_replace("/\[$key\]/", $value, $displayed_string);
      }

      // Create a clickable link to the corresponding entity when one exists.
      $renderable_item = $lookup_manager->getRenderableItem($displayed_string, $values['entity_id']);

      $list[$delta] = $renderable_item;
    }

    // Will convert $list to a markup list if there is more than one item.
    $elements = $this->createListMarkup($list);
    return $elements;
  }

}
