<?php

namespace Drupal\tripal_chado\Plugin\Field\FieldType;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tripal\TripalField\Attribute\TripalFieldType;
use Drupal\tripal_chado\TripalField\ChadoFieldItemBase;
use Drupal\tripal_chado\TripalStorage\ChadoBoolStoragePropertyType;
use Drupal\tripal_chado\TripalStorage\ChadoIntStoragePropertyType;
use Drupal\tripal_chado\TripalStorage\ChadoTextStoragePropertyType;
use Drupal\tripal\Entity\TripalEntityType;

/**
 * Plugin implementation of Default Tripal field for sequence coordinates.
 */
#[TripalFieldType(
  id: 'chado_sequence_coordinates_default',
  category: 'tripal_chado',
  label: new TranslatableMarkup('Chado Sequence Coordinates'),
  description: new TranslatableMarkup('Locations on reference sequences where the feature is located'),
  default_widget: 'chado_sequence_coordinates_widget_default',
  default_formatter: 'chado_sequence_coordinates_formatter_default',
  cardinality: 1,
)]
class ChadoSequenceCoordinatesDefault extends ChadoFieldItemBase {

  public static $id = "chado_sequence_coordinates_default";

  /**
   * {@inheritdoc}
   */
  public static function mainPropertyName() {
    return 'sequence_coordinates';
  }

  /**
  * {@inheritdoc}
  */
  public static function defaultFieldSettings() {
    $settings = parent::defaultFieldSettings();
    $settings['termIdSpace'] = 'data';
    $settings['termAccession'] = '2012';
    return $settings;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultStorageSettings() {
    $settings = parent::defaultStorageSettings();
    $settings['storage_plugin_settings']['base_table'] = 'feature';
    return $settings;
  }

  /**
   * {@inheritdoc}
   */
  public static function tripalTypes($field_definition) {

    // Create variables for easy access to settings.
    $entity_type_id = $field_definition->getTargetEntityTypeId();

    $field_settings = $field_definition->getSetting('storage_plugin_settings');
    $base_table = $field_settings['base_table'];

    // If we don't have a base table then we're not ready to specify the
    // properties for this field.
    if (!$base_table) {
      return;
    }

    // Get the property terms by using the Chado table columns they map to.

    $ft_uniqname_term = self::getColumnTermId('feature', 'name', 'schema:name');

    $feature_id_term = self::getColumnTermId('featureloc', 'feature_id', 'SO:0000110');
    $srcfeature_id_term = self::getColumnTermId('featureloc', 'srcfeature_id', 'data:3002');
    $fmin_term = self::getColumnTermId('featureloc', 'fmin', 'local:fmin');
    $is_fmin_partial_term = self::getColumnTermId('featureloc', 'is_fmin_partial', 'local:is_fmin_partial');
    $fmax_term = self::getColumnTermId('featureloc', 'fmax', 'local:fmax');
    $is_fmax_partial_term = self::getColumnTermId('featureloc', 'is_fmax_partial', 'local:is_fmax_partial');
    $strand_term = self::getColumnTermId('featureloc', 'strand', 'data:0853');
    $phase_term = self::getColumnTermId('featureloc', 'phase', 'data:2336');
    $residue_info_term = self::getColumnTermId('featureloc', 'residue_info', 'local:residue_info');
    $locgroup_term = self::getColumnTermId('featureloc', 'locgroup', 'local:locgroup');
    $rank_term = self::getColumnTermId('featureloc', 'rank', 'OBCS:0000117');

    // Get property terms using Chado table columns they map to. Return the properties for this field.
    $properties = [];

    $properties[] = new ChadoIntStoragePropertyType($entity_type_id, self::$id, 'record_id', self::$record_id_term, [
      'action' => 'store_id',
      'drupal_store' => TRUE,
      'path' => 'feature.feature_id',
    ]);

    $properties[] = new ChadoIntStoragePropertyType($entity_type_id, self::$id, 'featureloc_id', self::$record_id_term, [
      'action' => 'store_pkey',
      'drupal_store' => TRUE,
      'path' => 'featureloc.featureloc_id',
    ]);

    $properties[] = new ChadoIntStoragePropertyType($entity_type_id, self::$id, 'fkey', $feature_id_term, [
      'action' => 'store_link',
      'drupal_store' => TRUE,
      'path' => 'feature.feature_id>featureloc.feature_id',
    ]);

    $properties[] = new ChadoIntStoragePropertyType($entity_type_id, self::$id, 'srcfeature_id', $srcfeature_id_term, [
      'action' => 'store',
      'path' => 'feature.feature_id>featureloc.feature_id;srcfeature_id',
    ]);

    $properties[] = new ChadoTextStoragePropertyType($entity_type_id, self::$id, 'uniquename', $ft_uniqname_term, [
      'action' => 'read_value',
      'path' => 'feature.feature_id>featureloc.feature_id;featureloc.srcfeature_id>feature.feature_id;uniquename',
    ]);

    $properties[] = new ChadoIntStoragePropertyType($entity_type_id, self::$id, 'fmin', $fmin_term, [
      'action' => 'store',
      'path' => 'feature.feature_id>featureloc.feature_id;fmin',
    ]);

    $properties[] = new ChadoBoolStoragePropertyType($entity_type_id, self::$id, 'is_fmin_partial', $is_fmin_partial_term, [
      'action' => 'store',
      'path' => 'feature.feature_id>featureloc.feature_id;is_fmin_partial',
    ]);

    $properties[] = new ChadoIntStoragePropertyType($entity_type_id, self::$id, 'fmax', $fmax_term, [
      'action' => 'store',
      'path' => 'feature.feature_id>featureloc.feature_id;fmax',
    ]);

    $properties[] = new ChadoBoolStoragePropertyType($entity_type_id, self::$id, 'is_fmax_partial', $is_fmax_partial_term, [
      'action' => 'store',
      'path' => 'feature.feature_id>featureloc.feature_id;is_fmax_partial',
    ]);

    $properties[] = new ChadoIntStoragePropertyType($entity_type_id, self::$id, 'strand', $strand_term, [
      'action' => 'store',
      'path' => 'feature.feature_id>featureloc.feature_id;strand',
    ]);

    $properties[] = new ChadoIntStoragePropertyType($entity_type_id, self::$id, 'phase', $phase_term, [
      'action' => 'store',
      'path' => 'feature.feature_id>featureloc.feature_id;phase',
    ]);

    $properties[] = new ChadoTextStoragePropertyType($entity_type_id, self::$id, 'residue_info', $residue_info_term, [
      'action' => 'store',
      'path' => 'feature.feature_id>featureloc.feature_id;residue_info',
    ]);

    $properties[] = new ChadoIntStoragePropertyType($entity_type_id, self::$id, 'locgroup', $locgroup_term, [
      'action' => 'store',
      'path' => 'feature.feature_id>featureloc.feature_id;locgroup',
    ]);

    $properties[] = new ChadoIntStoragePropertyType($entity_type_id, self::$id, 'rank', $rank_term, [
      'action' => 'store',
      'path' => 'feature.feature_id>featureloc.feature_id;rank',
    ]);

    return($properties);
  }

  /**
   * {@inheritDoc}
   * @see \Drupal\tripal_chado\TripalField\ChadoFieldItemBase::isCompatible()
   */
  public function isCompatible(TripalEntityType $entity_type) : bool {
    $compatible = FALSE;

    // Get the base table for the content type.
    $base_table = $entity_type->getThirdPartySetting('tripal', 'chado_base_table');

    // This is a "specialty" field for a single content type
    if ($base_table == 'feature') {
      $compatible = TRUE;
    }
    return $compatible;
  }

}
