<?php

namespace Drupal\Tests\tripal_chado\Functional\Service;

use Drupal\Tests\tripal_chado\Functional\ChadoTestBrowserBase;
use Drupal\tripal\TripalVocabTerms\TripalTerm;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the TripalPublish service in the context of the Chado content types.
 *
 * @group Tripal
 * @group Tripal Content
 */
#[Group('Tripal')]
#[Group('Tripal Content')]
class ChadoTripalPublishTest extends ChadoTestBrowserBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['user', 'path', 'tripal', 'tripal_chado'];

  /**
   * A helper function for adding an organism record to Chado.
   *
   * @param \Drupal\tripal\TripalDBX\TripalDbxConnection $chado
   *   A chado database object.
   * @param array $details
   *   The key/value pairs of entries for the organism. The keys correspond
   *   to the columns of the organism table.
   *
   * @return int
   *   The organism_id.
   */
  public function addChadoOrganism($chado, $details) {

    $insert = $chado->insert('1:organism');
    $insert->fields([
      'genus' => $details['genus'],
      'species' => $details['species'],
      'type_id' => array_key_exists('type_id', $details) ? $details['type_id'] : NULL,
      'infraspecific_name' => array_key_exists('infraspecific_name', $details) ? $details['infraspecific_name'] : NULL,
      'abbreviation' => array_key_exists('abbreviation', $details) ? $details['abbreviation'] : NULL,
      'common_name' => array_key_exists('common_name', $details) ? $details['common_name'] : NULL,
      'comment' => array_key_exists('comment', $details) ? $details['comment'] : NULL,
    ]);
    return $insert->execute();
  }

  /**
   * A helper function for adding a project record to Chado.
   *
   * @param \Drupal\tripal\TripalDBX\TripalDbxConnection $chado
   *   A chado database object.
   * @param array $details
   *   The key/value pairs of entries for the project. The keys correspond
   *   to the columns of the project table.
   *
   * @return int
   *   The project_id.
   */
  public function addChadoProject($chado, $details) {
    $insert = $chado->insert('1:project');
    $insert->fields([
      'name' => $details['name'],
      'description' => array_key_exists('description', $details) ? $details['description'] : NULL,
    ]);
    return $insert->execute();
  }

  /**
   * A helper function for adding a contact record to Chado.
   *
   * @param \Drupal\tripal\TripalDBX\TripalDbxConnection $chado
   *   A chado database object.
   * @param array $details
   *   The key/value pairs of entries for the contact. The keys correspond
   *   to the columns of the contact table.
   *
   * @return int
   *   The contact_id.
   */
  public function addChadoContact($chado, $details) {
    $insert = $chado->insert('1:contact');
    $insert->fields([
      'name' => $details['name'],
      'type_id' => array_key_exists('type_id', $details) ? $details['type_id'] : NULL,
      'description' => array_key_exists('description', $details) ? $details['description'] : NULL,
    ]);
    return $insert->execute();
  }

  /**
   * A helper function for adding a project_contact record to Chado.
   *
   * @param \Drupal\tripal\TripalDBX\TripalDbxConnection $chado
   *   A chado database object.
   * @param array $details
   *   The key/value pairs of entries for the project_contact. The keys
   *   correspond to the columns of the project_contact table.
   *
   * @return int
   *   The project_contact_id.
   */
  public function addChadoProjectContact($chado, $details) {
    $insert = $chado->insert('1:project_contact');
    $insert->fields([
      'project_id' => $details['project_id'],
      'contact_id' => $details['contact_id'],
    ]);
    return $insert->execute();
  }

  /**
   * A helper function for adding a stock record to Chado.
   *
   * @param \Drupal\tripal\TripalDBX\TripalDbxConnection $chado
   *   A chado database object.
   * @param array $details
   *   The key/value pairs of entries for the stock. The keys correspond
   *   to the columns of the stock table.
   *
   * @return int
   *   The stock_id.
   */
  public function addChadoStock($chado, $details) {
    $insert = $chado->insert('1:stock');
    $insert->fields([
      'dbxref_id' => array_key_exists('dbxref_id', $details) ? $details['dbxref_id'] : NULL,
      'organism_id' => array_key_exists('organism_id', $details) ? $details['organism_id'] : NULL,
      'name' => array_key_exists('name', $details) ? $details['name'] : NULL,
      'uniquename' => $details['uniquename'],
      'description' => array_key_exists('description', $details) ? $details['description'] : NULL,
      'type_id' => $details['type_id'],
      'is_obsolete' => array_key_exists('is_obsolete', $details) ? $details['is_obsolete'] : 0,
    ]);
    return $insert->execute();
  }

  /**
   * A helper function for adding a project_stock record to Chado.
   *
   * @param \Drupal\tripal\TripalDBX\TripalDbxConnection $chado
   *   A chado database object.
   * @param array $details
   *   The key/value pairs of entries for the project_stock. The keys correspond
   *   to the columns of the project_stock table.
   *
   * @return int
   *   The project_stock_id.
   */
  public function addChadoProjectStock($chado, $details) {
    $insert = $chado->insert('1:project_stock');
    $insert->fields([
      'project_id' => $details['project_id'],
      'stock_id' => $details['stock_id'],
    ]);
    return $insert->execute();
  }

  /**
   * A helper function for adding an array design record to Chado.
   *
   * @param \Drupal\tripal\TripalDBX\TripalDbxConnection $chado
   *   A chado database object.
   * @param array $details
   *   The key/value pairs of entries for the array design. The keys correspond
   *   to the columns of the arraydesign table.
   *
   * @return int
   *   The arraydesign_id.
   */
  public function addChadoArrayDesign($chado, $details) {
    $insert = $chado->insert('1:arraydesign');
    $insert->fields([
      'name' => $details['name'],
      'manufacturer_id' => array_key_exists('manufacturer_id', $details) ? $details['manufacturer_id'] : 1,
      'platformtype_id' => array_key_exists('platformtype_id', $details) ? $details['platformtype_id'] : 1,
      'num_of_elements' => array_key_exists('num_of_elements', $details) ? $details['num_of_elements'] : NULL,
    ]);
    return $insert->execute();
  }

  /**
   * A helper function for adding a property to a record in Chado.
   *
   * @param \Drupal\tripal\TripalDBX\TripalDbxConnection $chado
   *   A chado database object.
   * @param string $base_table
   *   The base table to which the property should be added.
   * @param array $details
   *   The key/value pairs of entries for the property. The keys correspond
   *   to the columns of the property table.
   *
   * @return int
   *   The property primary key.
   */
  public function addProperty($chado, $base_table, $details) {

    $insert = $chado->insert('1:' . $base_table . 'prop');
    $insert->fields([
      $base_table . '_id' => $details[$base_table . '_id'],
      'value' => $details['value'],
      'type_id' => $details['type_id'],
      'rank' => $details['rank'],
    ]);
    return $insert->execute();
  }

  /**
   * A helper function to test if the elements of a field item are present.
   *
   * @param string $bundle
   *   The content type bundle name (e.g. 'organism').
   * @param string $field_name
   *   The name of the field that should be queried.
   * @param int $num_expected
   *   The number of items that are expected to be found when applying the
   *   conditions specified in the $match argument.
   * @param array $match
   *   An array of key/value pairs where the keys are the column names of
   *   field table in Drupal and the values are those to match in a select
   *   condition.  All fields other than the 'entity_id', 'bundle', 'delta'
   *   'deleted',  'langcode', and 'revision' have the field name as a prefix.
   *   But the keys need not include the prefix, just the field property key.
   *   The field name prefix will be added automatically.
   * @param array $check
   *   An array of key/value pairs where the keys are the column names of the
   *   field table in Drupal and the values are checked that they match
   *   what is in the table. The same rules apply for the key naming as in
   *   the $match argument.
   */
  public function checkFieldItem($bundle, $field_name, $num_expected, $match, $check) {

    $drupal_columns = ['bundle', 'entity_id', 'revision' ,'delta', 'deleted', 'langcode'];

    $public = \Drupal::service('database');
    $select = $public->select('tripal_entity__' . $field_name, 'f');
    $select->fields('f');
    $select->condition('bundle', $bundle);
    foreach ($match as $key => $val) {
      $column_name = $key;
      if (!in_array($key, $drupal_columns)) {
        $column_name = $field_name . '_' . $key;
      }
      $select->condition($column_name, $val);
    }
    $select->orderBy('delta');
    $result = $select->execute();
    $records = $result->fetchAll();

    $this->assertCount($num_expected, $records,
        'The number of items expected for field "' . $field_name .'" with bundle "'
        . $bundle . '" is not correct.');

    foreach ($records as $delta => $record) {

      // Make sure we have an entity ID for the specified record.
      $this->assertNotNull($record->entity_id,
        'The entity_id for a published item is missing for the field "'
          . $field_name . '" at delta ' . $delta);

      // Make sure the expected values are present.
      foreach ($check as $key => $val) {
        $column_name = $key;
        if (!in_array($key, $drupal_columns)) {
          $column_name = $field_name . '_' . $key;
        }
        $this->assertEquals($val, $record->$column_name,
          'The value for, "' . $column_name . '", is not correct what we expected.');
      }
    }
  }

  /**
   * A helper function to add a stock field to the project content type.
   * Here we use a finite cardinality of 15.
   */
  public function attachProjectStockField() {
    /** @var \Drupal\tripal\Services\TripalFieldCollection $fields_service **/
    $fields_service = \Drupal::service('tripal.tripalfield_collection');
    $stock_field = [
      'name' => 'project_stock',
      'content_type' => 'project',
      'label' => 'Germplasm',
      'type' => 'chado_stock_type_default',
      'description' => "Germplasm related to this project.",
      'cardinality' => 15,
      'required' => FALSE,
      'storage_settings' => [
        'storage_plugin_id' => 'chado_storage',
        'storage_plugin_settings'=> [
          'base_table' => 'project',
          'linker_table' => 'project_stock',
          'linker_fkey_column' => 'stock_id',
        ],
      ],
      'settings' => [
        'termIdSpace' => 'NCIT',
        'termAccession' => 'C70699',
      ],
      'display' => [
        'view' => [
          'default' => [
            'region' => 'content',
            'label' => 'above',
            'weight' => 15
          ],
        ],
        'form' => [
          'default'=> [
            'region'=> 'content',
            'weight' => 15
          ],
        ],
      ],
    ];
    $reason = '';
    $is_valid = $fields_service->validate($stock_field, $reason);
    $this->assertTrue($is_valid, $reason);
    $is_added = $fields_service->addBundleField($stock_field);
    $this->assertTrue($is_added, 'The stock field could not be added to project.');
  }

  /**
   * A helper function to add fields to the organism content types.
   */
  public function attachOrganismPropertyFields() {

    /** @var \Drupal\tripal\Services\TripalFieldCollection $fields_service **/
    // Now add a ChadoProperty field for the two types of properties.
    $fields_service = \Drupal::service('tripal.tripalfield_collection');
    $prop_field1 = [
      'name' => 'field_note',
      'content_type' => 'organism',
      'label' => 'Note',
      'type' => 'chado_property_type_default',
      'description' => "A note about this organism.",
      'cardinality' => -1,
      'required' => FALSE,
      'storage_settings' => [
        'storage_plugin_id' => 'chado_storage',
        'storage_plugin_settings' => [
          'base_table' => 'organism',
          'prop_table' => 'organismprop'
        ],
      ],
      'settings' => [
        'termIdSpace' => 'local',
        'termAccession' => "Note",
      ],
      'display' => [
        'view' => [
          'default' => [
            'region' => 'content',
            'label' => 'above',
            'weight' => 15
          ],
        ],
        'form' => [
          'default' => [
            'region' => 'content',
            'weight' => 15
          ],
        ],
      ],
    ];
    $reason = '';
    $is_valid = $fields_service->validate($prop_field1, $reason);
    $this->assertTrue($is_valid, $reason);
    $is_added = $fields_service->addBundleField($prop_field1);
    $this->assertTrue($is_added, 'The organism property field "local:Note" could not be added.');

    // Now add a ChadoProperty field for the two types of properties.
    $prop_field2 = [
      'name' => 'field_comment',
      'content_type' => 'organism',
      'label' => 'Comment',
      'type' => 'chado_property_type_default',
      'description' => "A comment about this organism.",
      'cardinality' => -1,
      'required' => FALSE,
      'storage_settings' => [
        'storage_plugin_id' => 'chado_storage',
        'storage_plugin_settings' => [
          'base_table' => 'organism',
          'prop_table' => 'organismprop'
        ],
      ],
      'settings' => [
        'termIdSpace' => 'schema',
        'termAccession' => "comment",
      ],
      'display' => [
        'view' => [
          'default' => [
            'region' => 'content',
            'label' => 'above',
            'weight' => 15
          ],
        ],
        'form' => [
          'default' => [
            'region' => 'content',
            'weight' => 15
          ],
        ],
      ],
    ];
    $reason = '';
    $is_valid = $fields_service->validate($prop_field2, $reason);
    $this->assertTrue($is_valid, $reason);
    $is_added = $fields_service->addBundleField($prop_field2);
    $this->assertTrue($is_added,
        'The Organism property field "schema:comment" could not be added.');
  }

  /**
   * Tests the TripalContentTypes class public functions.
   */
  public function testChadoTripalPublishService() {

    // The behavior depends on settings made in TripalEntitySettingsForm
    // Depending on default_cache_backend_field_values the values are either
    // saved in the entity or not.
    $config_edit = \Drupal::configFactory()->getEditable('tripal.settings');

    // Prepare Chado.
    $chado = $this->createTestSchema(ChadoTestBrowserBase::PREPARE_TEST_CHADO);

    // Add the CV terms. These normally get added during a prepare and
    // the Chado schema is prepared but not Drupal schema and it needs to
    // know about the terms used for content types and fields.
    $terms_setup = \Drupal::service('tripal_chado.terms_init');
    $terms_setup->installTerms();

    // Create the terms for the field property storage types.
    /** @var \Drupal\tripal\TripalVocabTerms\PluginManagers\TripalIdSpaceManager $idsmanager */
    $idsmanager = \Drupal::service('tripal.collection_plugin_manager.idspace');

    $local_db = $idsmanager->loadCollection('local', "chado_id_space");
    $note_term = new TripalTerm();
    $note_term->setName('Note');
    $note_term->setIdSpace('local');
    $note_term->setVocabulary('local');
    $note_term->setAccession('Note');
    $local_db->saveTerm($note_term);

    $schema_db = $idsmanager->loadCollection('schema', "chado_id_space");
    $comment_term = new TripalTerm();
    $comment_term->setName('comment');
    $comment_term->setIdSpace('schema');
    $comment_term->setVocabulary('schema');
    $comment_term->setAccession('comment');
    $schema_db->saveTerm($comment_term);

    // Make sure we have the content types and fields that we want to test.
    $collection_ids = ['general_chado', 'expression_chado'];
    $content_type_setup = \Drupal::service('tripal.tripalentitytype_collection');
    $publish_service = \Drupal::service('tripal.backend_publish');
    $chado_publish = $publish_service->createInstance('chado_storage', []);
    $fields_setup = \Drupal::service('tripal.tripalfield_collection');
    $content_type_setup->install($collection_ids);
    $fields_setup->install($collection_ids);
    $this->attachProjectStockField();

    // %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    // % Run the first test set with caching disabled. %
    // %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    // Update configuration.
    $config_edit->set('tripal_entity_type.default_cache_backend_field_values', FALSE);
    $config_edit->save();

    //
    // Test publishing when no records are available.
    //
    $entities = $chado_publish->publish(['bundle' => 'organism', 'datastore' => 'chado_storage']);
    $this->assertTrue(count($entities) == 0,
      'The TripalPublish service should return 0 entities when no records are available.');

    //
    // Test publishing a single record.
    //
    $taxrank_db = $idsmanager->loadCollection('TAXRANK', "chado_id_space");
    $subspecies_term_id = $taxrank_db->getTerm('0000023')->getInternalId();

    $organism_id = $this->addChadoOrganism($chado, [
      'genus' => 'Oryza',
      'species' => 'species',
      'abbreviation' => 'O. sativa',
      'type_id' => $subspecies_term_id,
      'infraspecific_name' => 'Japonica',
      'comment' => 'rice is nice'
    ]);
    $entities = $chado_publish->publish(['bundle' => 'organism', 'datastore' => 'chado_storage']);
    $this->assertCount(1, $entities,
      'The TripalPublish service should have published 1 organism.');
    $entity_id = array_key_first($entities);

    // Test that entries were added for all field items and that fields that
    // shouldn't be saved in Drupal are equal to the field's default value.
    $this->checkFieldItem('organism', 'organism_genus', 1,
        ['record_id' => $organism_id],
        ['bundle' => 'organism', 'entity_id' => $entity_id, 'value' => '']);

    $this->checkFieldItem('organism', 'organism_species', 1,
        ['record_id' => $organism_id],
        ['bundle' => 'organism', 'entity_id' => $entity_id, 'value' => '']);

    $this->checkFieldItem('organism', 'organism_abbreviation', 1,
        ['record_id' => $organism_id],
        ['bundle' => 'organism', 'entity_id' => $entity_id, 'value' => '']);

    $this->checkFieldItem('organism', 'organism_infraspecific_name', 1,
        ['record_id' => $organism_id],
        ['bundle' => 'organism', 'entity_id' => $entity_id, 'value' => '']);

    $this->checkFieldItem('organism', 'organism_infraspecific_type', 1,
        ['record_id' => $organism_id],
        ['bundle' => 'organism', 'entity_id' => $entity_id, 'type_id' => 0]);

    $this->checkFieldItem('organism', 'organism_comment', 1,
        ['record_id' => $organism_id],
        ['bundle' => 'organism', 'entity_id' => $entity_id, 'value' => '']);

    // Test that the title via token replacement is working.
    $this->assertTrue($entities[$entity_id] == '<em>Oryza species</em> subspecies <em>Japonica</em>',
        'The title of a Chado organism is incorrect after publishing: ' . array_values($entities)[0] . '!=' . '<em>Oryza species</em> subspecies <em>Japonica</em>');

    //
    // Test a second entity. Also use a title without all tokens.
    //
    $organism_id2 = $this->addChadoOrganism($chado, [
      'genus' => 'Gorilla',
      'species' => 'gorilla',
      'abbreviation' => 'G. gorilla',
      'comment' => 'Gorilla'
    ]);
    $entities = $chado_publish->publish(['bundle' => 'organism', 'datastore' => 'chado_storage']);
    $this->assertCount(1, $entities,
      'The TripalPublish service should have published 1 organism.');
    $entity_id = array_key_first($entities);
    // Token parser will also remove a token if it is empty, e.g. infraspecific
    // nomenclature absent.
    $this->assertEquals('<em>Gorilla gorilla</em>', $entities[$entity_id],
        'The title of Chado organism with missing tokens is incorrect after publishing');

    $this->checkFieldItem('organism', 'organism_genus', 1,
        ['record_id' => $organism_id2],
        ['bundle' => 'organism', 'entity_id' => $entity_id, 'value' => '']);

    $this->checkFieldItem('organism', 'organism_species', 1,
        ['record_id' => $organism_id2],
        ['bundle' => 'organism', 'entity_id' => $entity_id, 'value' => '']);

    $this->checkFieldItem('organism', 'organism_abbreviation', 1,
        ['record_id' => $organism_id2],
        ['bundle' => 'organism', 'entity_id' => $entity_id, 'value' => '']);

    $this->checkFieldItem('organism', 'organism_comment', 1,
        ['record_id' => $organism_id2],
        ['bundle' => 'organism', 'entity_id' => $entity_id, 'value' => '']);

    // We expect no infraspecies here, so expected count is zero.
    $this->checkFieldItem('organism', 'organism_infraspecific_name', 0,
        ['record_id' => $organism_id2],
        []);

    $this->checkFieldItem('organism', 'organism_infraspecific_type', 0,
        ['record_id' => $organism_id2],
        []);

    //
    // Test publishing properties.
    //
    $comment_type_id = $schema_db->getTerm('comment')->getInternalId();
    $note_type_id = $local_db->getTerm('Note')->getInternalId();
    $this->attachOrganismPropertyFields();
    $this->addProperty($chado, 'organism', [
      'organism_id' => $organism_id,
      'type_id' => $note_type_id,
      'value' => 'Note 1',
      'rank' => 1,
    ]);
    $this->addProperty($chado, 'organism', [
      'organism_id' => $organism_id,
      'type_id' => $note_type_id,
      'value' => 'Note 0',
      'rank' => 0,
    ]);
    $this->addProperty($chado, 'organism', [
      'organism_id' => $organism_id,
      'type_id' => $note_type_id,
      'value' => 'Note 2',
      'rank' => 2,
    ]);

    $this->addProperty($chado, 'organism', [
      'organism_id' => $organism_id,
      'type_id' => $comment_type_id,
      'value' => 'Comment 0',
      'rank' => 0,
    ]);
    $this->addProperty($chado, 'organism', [
      'organism_id' => $organism_id,
      'type_id' => $comment_type_id,
      'value' => 'Comment 1',
      'rank' => 1,
    ]);

    // Now publish the organism content type again.
    $entities = $chado_publish->publish(['bundle' => 'organism', 'datastore' => 'chado_storage', 'republish' => 1]);

    // Because we added properties for the first organism we should get its
    // entity in those returned, but not the gorilla organism.
    $this->assertEquals('<em>Oryza species</em> subspecies <em>Japonica</em>', array_values($entities)[0],
      'The Oryza species subspecies Japonica organism should appear in the published list because it has new properties.');
    $this->assertCount(1, array_values($entities),
      'There should only be one published entity for a single organism with new properties.');

    // Check that the property values got published. The type_id should be
    // 0 in the drupal field table, the default for an integer.
    $this->checkFieldItem('organism', 'field_note', 1,
        ['record_id' => $organism_id, 'prop_id' => 1],
        ['type_id' => 0, 'linker_id' => $organism_id,
         'bundle' => 'organism', 'entity_id' => 1]);

    $this->checkFieldItem('organism', 'field_note', 1,
        ['record_id' => $organism_id, 'prop_id' => 2],
        ['type_id' => 0, 'linker_id' => $organism_id,
         'bundle' => 'organism', 'entity_id' => 1]);

    $this->checkFieldItem('organism', 'field_note', 1,
        ['record_id' => $organism_id, 'prop_id' => 3],
        ['type_id' => 0, 'linker_id' => $organism_id,
         'bundle' => 'organism', 'entity_id' => 1]);

    $this->checkFieldItem('organism', 'field_comment', 1,
        ['record_id' => $organism_id, 'prop_id' => 4],
        ['type_id' => 0, 'linker_id' => $organism_id,
         'bundle' => 'organism', 'entity_id' => 1]);

    $this->checkFieldItem('organism', 'field_comment', 1,
        ['record_id' => $organism_id, 'prop_id' => 5],
        ['type_id' => 0, 'linker_id' => $organism_id,
         'bundle' => 'organism', 'entity_id' => 1]);

    // Check that only the exact number of properties were published.
    $this->checkFieldItem('organism', 'field_note', 3, ['entity_id' => 1], []);
    $this->checkFieldItem('organism', 'field_comment', 2, ['entity_id' => 1], []);

    //
    // Test publishing a field that uses a linker table.
    //

    // Create and publish the contacts and the project.
    $contact_db = $idsmanager->loadCollection('TCONTACT', "chado_id_space");
    $person_term_id = $contact_db->getTerm('0000003')->getInternalId();
    $contact_id1 = $this->addChadoContact($chado, [
      'name' => 'John Doe',
      'type_id' => $person_term_id,
      'description' => 'Bioinformaticist extrodinaire'
    ]);
    $contact_id2 = $this->addChadoContact($chado, [
      'name' => 'Lady Gaga',
      'type_id' => $person_term_id,
      'description' => 'Pop star'
    ]);
    $project_id1 = $this->addChadoProject($chado, [
      'name' => 'Bad Project',
      'description' => 'Want your bad project'
    ]);
    $project_id2 = $this->addChadoProject($chado, [
      'name' => 'Project Face',
      'description' => 'I wanna project like they do in Texas, please'
    ]);
    $project_contact_id1 = $this->addChadoProjectContact($chado, [
      'project_id' => $project_id1,
      'contact_id' => $contact_id1,
    ]);
    $project_contact_id2 = $this->addChadoProjectContact($chado, [
      'project_id' => $project_id1,
      'contact_id' => $contact_id2,
    ]);
    $project_contact_id3 = $this->addChadoProjectContact($chado, [
      'project_id' => $project_id2,
      'contact_id' => $contact_id2,
    ]);

    // Now publish the projects and contacts. We check that 3 items are
    // published because there is a null contact and currently there is
    // nothing to prevent that contact from being published. (Issue #1809)
    $entities = $chado_publish->publish(['bundle' => 'contact', 'datastore' => 'chado_storage']);
    $this->assertCount(3, $entities,
        'Failed to publish 3 contact entities.');

    $entities = $chado_publish->publish(['bundle' => 'project', 'datastore' => 'chado_storage']);
    $this->assertCount(2, $entities,
      'Failed to publish 2 project entities.');

    // Make sure that the linked records are also published for each project.
    // The chado_contact_type_default is the field we're testing got published.
    $this->checkFieldItem('project', 'project_contact', 1,
        ['record_id' => $project_id1, 'linker_id' => $project_contact_id1],
        ['link' => $project_id1, 'bundle' => 'project', 'entity_id' => 6, 'contact_id' => $contact_id1]);

    $this->checkFieldItem('project', 'project_contact', 1,
        ['record_id' => $project_id1, 'linker_id' => $project_contact_id2],
        ['link' => $project_id1, 'bundle' => 'project', 'entity_id' => 6, 'contact_id' => $contact_id2]);

    $this->checkFieldItem('project', 'project_contact', 1,
        ['record_id' => $project_id2, 'linker_id' => $project_contact_id3],
        ['link' => $project_id2, 'bundle' => 'project', 'entity_id' => 7, 'contact_id' => $contact_id2]);

    // Check that only the exact number of linked items were published.
    $this->checkFieldItem('project', 'project_contact', 2, ['entity_id' => 6], []);
    $this->checkFieldItem('project', 'project_contact', 1, ['entity_id' => 7], []);

    // Test publishing of integer fields.
    $array_design_id1 = $this->addChadoArrayDesign($chado, [
      'name' => 'AD1',
      'num_of_elements' => 1,
    ]);
    $entities = $chado_publish->publish(['bundle' => 'array_design', 'datastore' => 'chado_storage']);
    $this->assertCount(1, $entities,
        'Failed to publish 1 array design entitity.');
    $this->checkFieldItem('array_design', 'array_design_num_of_elements', 1,
        ['record_id' => $array_design_id1],
        ['bundle' => 'array_design', 'entity_id' => 8, 'value' => 0]);
    // We do not expect a NULL integer item to be published.
    $this->checkFieldItem('array_design', 'array_design_num_array_columns', 0,
        ['record_id' => $array_design_id1],
        []);

    // %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    // %     Run tests set with caching enabled.       %
    // %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    // Update configuration.
    $config_edit->set('tripal_entity_type.default_cache_backend_field_values', TRUE);
    $config_edit->save();
    //
    // Test publishing a single record: Organism 3.
    //
    $taxrank_db = $idsmanager->loadCollection('TAXRANK', "chado_id_space");
    $subspecies_term_id = $taxrank_db->getTerm('0000023')->getInternalId();

    $organism_id3 = $this->addChadoOrganism($chado, [
      'genus' => 'Oryza',
      'species' => 'sativa',
      'abbreviation' => 'O. sativa',
      'type_id' => $subspecies_term_id,
      'infraspecific_name' => 'Japonica',
      'comment' => 'rice is nice, too'
    ]);
    $entities = $chado_publish->publish(['bundle' => 'organism', 'datastore' => 'chado_storage']);
    $this->assertCount(1, $entities,
      'The TripalPublish service should have published 1 organism.');
    $entity_id = array_key_first($entities);

    // Test that entries were added for all field items and that fields that
    // shouldn't be saved in Drupal are equal to the field's default value.
    $this->checkFieldItem('organism', 'organism_genus', 1,
        ['record_id' => $organism_id3],
        ['bundle' => 'organism', 'entity_id' => $entity_id, 'value' => 'Oryza']);

    $this->checkFieldItem('organism', 'organism_species', 1,
        ['record_id' => $organism_id3],
        ['bundle' => 'organism', 'entity_id' => $entity_id, 'value' => 'sativa']);

    $this->checkFieldItem('organism', 'organism_abbreviation', 1,
        ['record_id' => $organism_id3],
        ['bundle' => 'organism', 'entity_id' => $entity_id, 'value' => 'O. sativa']);

    $this->checkFieldItem('organism', 'organism_infraspecific_name', 1,
        ['record_id' => $organism_id3],
        ['bundle' => 'organism', 'entity_id' => $entity_id, 'value' => 'Japonica']);

    $this->checkFieldItem('organism', 'organism_infraspecific_type', 1,
        ['record_id' => $organism_id3],
        ['bundle' => 'organism', 'entity_id' => $entity_id, 'type_id' => $subspecies_term_id]);

    $this->checkFieldItem('organism', 'organism_comment', 1,
        ['record_id' => $organism_id3],
        ['bundle' => 'organism', 'entity_id' => $entity_id, 'value' => 'rice is nice, too']);

    // Test that the title via token replacement is working.
    $this->assertTrue($entities[$entity_id] == '<em>Oryza sativa</em> subspecies <em>Japonica</em>',
        'The title of a Chado organism is incorrect after publishing: ' . array_values($entities)[0] . '!=<em>Oryza sativa</em> subspecies <em>Japonica</em>');

    // Because we added properties for the first organism we should get its
    // entity in those returned, but not the gorilla organism.
    $this->assertEquals(
      '<em>Oryza sativa</em> subspecies <em>Japonica</em>',
      array_values($entities)[0],
      'The Oryza species subspecies Japonica organism should appear in the published list because it has new properties.'
    );

    //
    // Test publishing properties.
    //
    $comment_type_id = $schema_db->getTerm('comment')->getInternalId();
    $note_type_id = $local_db->getTerm('Note')->getInternalId();
    $this->addProperty($chado, 'organism', [
      'organism_id' => $organism_id3,
      'type_id' => $note_type_id,
      'value' => 'Note 1',
      'rank' => 1,
    ]);
    $this->addProperty($chado, 'organism', [
      'organism_id' => $organism_id3,
      'type_id' => $note_type_id,
      'value' => 'Note 0',
      'rank' => 0,
    ]);
    $this->addProperty($chado, 'organism', [
      'organism_id' => $organism_id3,
      'type_id' => $note_type_id,
      'value' => 'Note 2',
      'rank' => 2,
    ]);

    $this->addProperty($chado, 'organism', [
      'organism_id' => $organism_id3,
      'type_id' => $comment_type_id,
      'value' => 'Comment 0',
      'rank' => 0,
    ]);
    $this->addProperty($chado, 'organism', [
      'organism_id' => $organism_id3,
      'type_id' => $comment_type_id,
      'value' => 'Comment 1',
      'rank' => 1,
    ]);

    // Now publish the organism content type again.
    $entities = $chado_publish->publish(['bundle' => 'organism', 'datastore' => 'chado_storage', 'republish' => TRUE]);
    $entity_id = array_key_first($entities);
    // Because we added properties for the first organism we should get its
    // entity in those returned, but not the gorilla organism.
    $this->assertEquals('<em>Oryza sativa</em> subspecies <em>Japonica</em>', array_values($entities)[0],
    'The Oryza species subspecies Japonica organism should appear in the published list because it has new properties.');
    $this->assertCount(1, array_values($entities),
    'There should only be one published entity for a single organism with new properties.');

    // Check that the property values got published. The type_id should be
    // be the $note_type_id in the drupal field table, not the default for an
    // integer.
    $this->checkFieldItem('organism', 'field_note', 1,
      ['record_id' => $organism_id3, 'prop_id' => 6],
      ['type_id' => $note_type_id, 'linker_id' => $organism_id3,
      'bundle' => 'organism', 'entity_id' => $entity_id]);

    $this->checkFieldItem('organism', 'field_note', 1,
      ['record_id' => $organism_id3, 'prop_id' => 7],
      ['type_id' => $note_type_id, 'linker_id' => $organism_id3,
      'bundle' => 'organism', 'entity_id' => $entity_id]);

    $this->checkFieldItem('organism', 'field_note', 1,
    ['record_id' => $organism_id3, 'prop_id' => 8],
    ['type_id' => $note_type_id, 'linker_id' => $organism_id3,
     'bundle' => 'organism', 'entity_id' => $entity_id]);

    $this->checkFieldItem('organism', 'field_comment', 1,
    ['record_id' => $organism_id3, 'prop_id' => 9],
    ['type_id' => $comment_type_id, 'linker_id' => $organism_id3,
     'bundle' => 'organism', 'entity_id' => $entity_id]);

    $this->checkFieldItem('organism', 'field_comment', 1,
    ['record_id' => $organism_id3, 'prop_id' => 10],
    ['type_id' => $comment_type_id, 'linker_id' => $organism_id3,
     'bundle' => 'organism', 'entity_id' => $entity_id]);

    // Check that only the exact number of properties were published.
    $this->checkFieldItem('organism', 'field_note', 3, ['entity_id' => 1], []);
    $this->checkFieldItem('organism', 'field_comment', 2, ['entity_id' => 1], []);

    //
    // Test a single entity: Organism 4. Also use a title without all tokens.
    //
    $organism_id4 = $this->addChadoOrganism($chado, [
      'genus' => 'Lepeophtheirus',
      'species' => 'salmonis',
      'common_name' => 'salmon louse',
      'abbreviation' => 'L. salmonis',
      'comment' => 'The salmon louse is a major ectoparasite of salmonids.'
    ]);
    $entities = $chado_publish->publish(['bundle' => 'organism', 'datastore' => 'chado_storage']);
    $this->assertCount(1, $entities,
      'The TripalPublish service should have published 1 organism.');
    $entity_id = array_key_first($entities);
    // Token parser will also remove a token if it is empty, e.g. infraspecific
    // nomenclature absent.
    $this->assertEquals('<em>Lepeophtheirus salmonis</em>', $entities[$entity_id],
        'The title of Chado organism with missing tokens is incorrect after publishing');

    $this->checkFieldItem('organism', 'organism_genus', 1,
        ['record_id' => $organism_id4],
        ['bundle' => 'organism', 'entity_id' => $entity_id, 'value' => 'Lepeophtheirus']);

    $this->checkFieldItem('organism', 'organism_species', 1,
        ['record_id' => $organism_id4],
        ['bundle' => 'organism', 'entity_id' => $entity_id, 'value' => 'salmonis']);

    $this->checkFieldItem('organism', 'organism_abbreviation', 1,
        ['record_id' => $organism_id4],
        ['bundle' => 'organism', 'entity_id' => $entity_id, 'value' => 'L. salmonis']);

    $this->checkFieldItem('organism', 'organism_common_name', 1,
        ['record_id' => $organism_id4],
        ['bundle' => 'organism', 'entity_id' => $entity_id, 'value' => 'salmon louse']);

    $this->checkFieldItem('organism', 'organism_comment', 1,
        ['record_id' => $organism_id4],
        ['bundle' => 'organism', 'entity_id' => $entity_id,
          'value' => 'The salmon louse is a major ectoparasite of salmonids.']);

    // We expect no infraspecies here, so expected count is zero.
    $this->checkFieldItem('organism', 'organism_infraspecific_name', 0,
        ['record_id' => $organism_id4],
        []);

    $this->checkFieldItem('organism', 'organism_infraspecific_type', 0,
        ['record_id' => $organism_id4],
        []);

    // Test publishing of integer fields.
    $array_design_id2 = $this->addChadoArrayDesign($chado, [
      'name' => 'AD2',
      'num_of_elements' => 12,
    ]);
    $entities = $chado_publish->publish(['bundle' => 'array_design', 'datastore' => 'chado_storage']);
    $entity_id = array_key_first($entities);
    $this->assertCount(1, $entities,
      'Failed to publish 1 array design entitity.');
    $this->checkFieldItem('array_design', 'array_design_num_of_elements', 1,
      ['record_id' => $array_design_id2],
      ['bundle' => 'array_design', 'entity_id' => $entity_id, 'value' => 12]);
    // We do not expect a NULL integer item to be published.
    $this->checkFieldItem('array_design', 'array_design_num_array_columns', 0,
      ['record_id' => $array_design_id2],
      []);

    // Test the max delta limit
    // Generate a large number of contact and stock records in chado,
    // and link every one to the first project.
    $new_contacts = [];
    $new_stocks = [];
    for ($i = 1; $i <= 120; $i++) {
      $contact_id = $this->addChadoContact($chado, [
        'name' => 'Additional Contact ' . $i,
        'description' => 'Numerous contact ' . $i,
      ]);
      $new_contacts[$i] = $contact_id;
      $stock_id = $this->addChadoStock($chado, [
        'name' => 'Additional Stock ' . $i,
        'uniquename' => 'stock_' . $i,
        'type_id' => 1,
        'description' => 'Numerous stock ' . $i,
      ]);
      $new_stocks[$i] = $stock_id;
      $project_contact_id = $this->addChadoProjectContact($chado, [
        'project_id' => $project_id1,
        'contact_id' => $contact_id,
      ]);
      $project_stock_id = $this->addChadoProjectStock($chado, [
        'project_id' => $project_id1,
        'stock_id' => $stock_id,
      ]);
    }

    // Note that the project_stock field has been added with cardinality 15.
    // Publish with inhibit set. No new contacts should
    // be published because the default global limit is exceeded.
    \Drupal::configFactory()
      ->getEditable('tripal.settings')
      ->set('tripal_entity_type.publish_global_max_delta_inhibit', 1)
      ->save();
    $entities = $chado_publish->publish(['bundle' => 'project', 'datastore' => 'chado_storage', 'republish' => TRUE]);
    // We should have published no contacts (2 were pre-existing).
    $this->checkFieldItem('project', 'project_contact', 2, ['entity_id' => 6], []);
    // We should have published no stocks.
    $this->checkFieldItem('project', 'project_stock', 0, ['entity_id' => 6], []);

    // Reset the inhibit setting, and publish using the default max delta
    // (i.e. 100). This tests where the global value has never been set.
    $config_edit->set('tripal_entity_type.publish_global_max_delta_inhibit', 0)->save();
    $entities = $chado_publish->publish(['bundle' => 'project', 'datastore' => 'chado_storage', 'republish' => TRUE]);
    // We should have published max delta + 1 contacts (2 were pre-existing)
    $this->checkFieldItem('project', 'project_contact', 101, ['entity_id' => 6], []);
    // We should have published cardinality + 1 stocks since cardinality
    // overrides max_delta.
    $this->checkFieldItem('project', 'project_stock', 16, ['entity_id' => 6], []);

    // Test setting a global max delta limit (increase default by 5)
    $config_edit->set('tripal_entity_type.publish_global_max_delta', 105)->save();
    $entities = $chado_publish->publish(['bundle' => 'project', 'datastore' => 'chado_storage', 'republish' => TRUE]);
    // We should have published max delta + 1 contacts (5 new,
    // 101 were pre-existing).
    $this->checkFieldItem('project', 'project_contact', 106, ['entity_id' => 6], []);
    // We should not have published any more stocks.
    $this->checkFieldItem('project', 'project_stock', 16, ['entity_id' => 6], []);
  }

}
