# Text-to-Speech (TTS) Module - Complete Documentation

**Machine Name**: `tts`  
**Version**: 1.0.1  
**Drupal**: 10.x | 11.x  
**AI Module**: 1.2.1+  
**Last Updated**: October 19, 2025

---

## Table of Contents

1. [Overview](#overview)
2. [Quick Start](#quick-start)
3. [Installation](#installation)
4. [Configuration](#configuration)
5. [Usage](#usage)
6. [AI Module Integration](#ai-module-integration)
7. [Technical Details](#technical-details)
8. [Troubleshooting](#troubleshooting)
9. [API Reference](#api-reference)

---

# Overview

## What is the TTS Module?

The Text-to-Speech module provides natural-sounding voice synthesis for Drupal content using AI-powered text-to-speech engines. It integrates seamlessly with the Drupal AI module to convert node content into high-quality audio.

## Features

- ✅ **Node Content Reading**: Automatically converts node pages into spoken audio
- ✅ **Natural AI Voices**: Uses AI-powered TTS engines for natural-sounding voices
- ✅ **Multiple Content Types**: Support for any Drupal content type
- ✅ **Configurable Settings**: Admin interface to configure TTS options
- ✅ **Audio Player Widget**: Embedded audio player on node pages
- ✅ **Accessibility**: Improves content accessibility for visually impaired users
- ✅ **REST API**: Programmatic access to generate TTS audio
- ✅ **Smart Caching**: Two-layer caching system with persistent file storage
- ✅ **Persistent Files**: Audio files survive cache clear operations 🎯
- ✅ **Multiple Formats**: MP3, OGG, WAV audio formats
- ✅ **Mobile Responsive**: Works perfectly on all devices
- ✅ **Automatic Cleanup**: Old audio files deleted on node update/delete (80% disk space savings)

## Requirements

- **Drupal**: 10.x or 11.x
- **AI Module**: 1.1.x or later (drupal/ai)
- **PHP**: 8.1 or higher
- **AI Provider**: At least one configured (OpenAI, Google Cloud, Azure, etc.)

## Supported AI Providers

| Provider | Models | Voices | Quality |
|----------|--------|--------|---------|
| **OpenAI** | tts-1, tts-1-hd | alloy, echo, fable, onyx, nova, shimmer | High |
| **Google Cloud** | WaveNet, Neural2 | 40+ languages | Very High |
| **Azure** | Neural voices | 100+ voices | High |
| **Amazon Polly** | Neural, Standard | 20+ languages | High |
| **IBM Watson** | Neural | 15+ languages | High |

---

# Quick Start

Get up and running in 5 minutes!

## Step 1: Install AI Module

```bash
composer require drupal/ai
ddev drush en ai -y
```

## Step 2: Enable TTS Module

```bash
ddev drush en tts -y
ddev drush cr
```

## Step 3: Configure AI Provider

**Option A: Via UI**
1. Go to `/admin/config/ai/settings`
2. Under "Text To Speech" section:
   - Provider: `openai`
   - Model: `tts-1-hd`
3. Save configuration

**Option B: Via Drush**
```bash
ddev drush config:set ai.settings default_providers.text_to_speech.provider_id openai
ddev drush config:set ai.settings default_providers.text_to_speech.model_id tts-1-hd
ddev drush cr
```

## Step 4: Configure TTS Settings

1. Go to `/admin/config/ai/tts`
2. Configure:
   - **Voice Model**: `alloy` (or nova, shimmer, etc.)
   - **Audio Format**: `mp3`
   - **Enabled Content Types**: Check Article, Page
   - **Included Fields**: `body`
3. Save configuration

## Step 5: Place Audio Player Block

1. Go to `/admin/structure/block`
2. Find your theme region (e.g., "Content")
3. Click "Place block"
4. Search for "TTS Audio Player"
5. Configure visibility:
   - Content types: Article, Page
   - Pages: `/node/*`
6. Save block

## Step 6: Test!

1. Create or visit an Article with text content
2. Audio player should appear
3. Click play to hear the audio! 🎵

---

# Installation

## Prerequisites

- Drupal 10.x or 11.x installed
- Composer for dependency management
- PHP 8.1 or higher
- API key for your chosen AI provider (OpenAI, Google Cloud, etc.)

## Installation Steps

### 1. Install AI Module

```bash
cd /path/to/drupal
composer require drupal/ai
ddev drush en ai -y
```

### 2. Install TTS Module

**Option A: Via Composer** (if published)
```bash
composer require drupal/tts
```

**Option B: Manual Installation**
```bash
# Place module in web/modules/custom/tts
cd web/modules/custom
# Copy tts folder here
```

### 3. Enable the Module

```bash
ddev drush en tts -y
ddev drush cr
```

### 4: Configure AI Provider

**For OpenAI**:
1. Get API key from https://platform.openai.com/api-keys
2. Go to `/admin/config/ai/settings`
3. Add OpenAI provider with your API key
4. Configure Text-to-Speech section:
   - Provider: `openai`
   - Model: `tts-1` or `tts-1-hd`

**For Google Cloud**:
1. Set up Google Cloud project
2. Enable Text-to-Speech API
3. Download service account JSON
4. Configure in AI module settings

**For Azure**:
1. Create Azure account
2. Create Cognitive Services resource
3. Get subscription key and region
4. Configure in AI module settings

### 5. Configure Module Settings

Go to `/admin/config/ai/tts`:

**Voice Settings**:
- Voice Model: `alloy` (OpenAI voices: alloy, echo, fable, onyx, nova, shimmer)

**Output Settings**:
- Audio Format: `mp3` (recommended)
- Max Text Length: `4096` characters

**Content Settings**:
- Enabled Content Types: Check Article, Page, etc.
- Included Fields: `body` (one per line)

**Cache Settings**:
- Cache Duration: `604800` seconds (7 days)

**Display Settings**:
- Show Player Block: ✅ Checked
- Player Position: `top` or `bottom`

### 6. Set Permissions

Go to `/admin/people/permissions`:

- **Anonymous user**: "Access TTS audio"
- **Authenticated user**: "Access TTS audio"
- **Content editor**: "Access TTS audio", "Generate TTS audio"
- **Administrator**: All TTS permissions

### 7. Place Audio Player Block

1. Go to `/admin/structure/block`
2. Find "Content" region
3. Place "TTS Audio Player" block
4. Configure visibility:
   - Content types: Article, Page
   - Pages: `/node/*`
5. Save

### 8. Verify Installation

```bash
# Check module status
ddev drush pm:list | grep tts

# Check configuration
ddev drush config:get ai.settings default_providers.text_to_speech

# Check for errors
ddev drush ws --count=10
```

---

# Configuration

## Configuration Locations

| Setting | Module | URL |
|---------|--------|-----|
| **Provider & Model** | AI Module | `/admin/config/ai/settings` |
| **API Keys** | AI Module | `/admin/config/ai/settings` |
| **Voice & Content** | TTS | `/admin/config/ai/tts` |
| **Block Placement** | Drupal Core | `/admin/structure/block` |
| **Permissions** | Drupal Core | `/admin/people/permissions` |

## AI Module Configuration

### Text-to-Speech Provider

Located at: `/admin/config/ai/settings`

**Configuration**:
- Scroll to "Text To Speech" section
- **Provider**: Select your AI provider (e.g., `openai`)
- **Model**: Select TTS model (e.g., `tts-1-hd`)

**Via Drush**:
```bash
ddev drush config:set ai.settings default_providers.text_to_speech.provider_id openai
ddev drush config:set ai.settings default_providers.text_to_speech.model_id tts-1-hd
```

## TTS Module Configuration

Located at: `/admin/config/ai/tts`

### Voice Settings

**Voice Model**: Provider-specific voice ID
- OpenAI: `alloy`, `echo`, `fable`, `onyx`, `nova`, `shimmer`
- Google Cloud: `en-US-Neural2-A`, `en-US-WaveNet-A`, etc.
- Azure: `en-US-AriaNeural`, etc.

### Output Settings

**Audio Format**: `mp3`, `ogg`, or `wav`
- Recommended: `mp3` (best compatibility)

**Max Text Length**: Maximum characters to process
- Default: `4096`
- Range: 100 - 100,000

### Content Settings

**Enabled Content Types**: Select which content types have TTS
- Check: Article, Page, or custom types

**Included Fields**: Fields to read (one per line)
```
body
field_summary
field_description
```

**Auto-generate on save**: ⚠️ May slow down node saves

### Cache Settings

**Cache Duration**: How long to cache audio files
- Default: `604800` seconds (7 days)
- Range: 3600 - 2592000 seconds

### Display Settings

**Show Player Block**: Enable/disable audio player
**Player Position**: `top` or `bottom` of content

## File System Configuration

Audio files are stored in: `public://tts/`

**Ensure directory is writable**:
```bash
mkdir -p web/sites/default/files/tts
chmod 755 web/sites/default/files/tts
```

---

# Usage

## For Site Visitors

1. Navigate to any enabled node page (Article, Page, etc.)
2. Look for "Listen to this page" audio player
3. Click play button to hear the content
4. Use controls:
   - Play/Pause
   - Seek through audio
   - Adjust volume
   - Download audio

### Keyboard Shortcuts

- `Space` or `K` - Play/Pause
- `←` Left Arrow - Rewind 5 seconds
- `→` Right Arrow - Fast forward 5 seconds
- `↑` Up Arrow - Increase volume
- `↓` Down Arrow - Decrease volume
- `M` - Mute/Unmute

## For Developers

### Programmatic Usage

```php
<?php

// Get TTS service
$tts_service = \Drupal::service('tts.text_to_speech');

// Generate audio from text
$text = "Hello, this is a test of text-to-speech.";
$audio_data = $tts_service->generateSpeech($text);

if ($audio_data) {
  $audio_url = $audio_data['url'];
  $duration = $audio_data['duration'];
  $format = $audio_data['format'];
}

// Generate audio from node
$node = \Drupal\node\Entity\Node::load(123);
$node_audio = $tts_service->generateNodeSpeech($node);

if ($node_audio) {
  $audio_url = $node_audio['url'];
  // Use the URL...
}

// Get audio URL directly
$audio_url = $tts_service->getAudioUrl($node);
```

### Custom Integration

**In your template**:
```php
<?php
$tts_service = \Drupal::service('tts.text_to_speech');
$variables['audio_url'] = $tts_service->getAudioUrl($node);
?>
```

**In Twig**:
```twig
{% if audio_url %}
  <audio controls>
    <source src="{{ audio_url }}" type="audio/mpeg">
    Your browser does not support the audio element.
  </audio>
{% endif %}
```

### REST API

#### Generate TTS for Node
```bash
GET /tts/node/{node_id}/audio
```

**Example**:
```bash
curl https://yoursite.com/tts/node/123/audio
```

**Response**: Audio file stream

#### Generate TTS from Text
```bash
POST /tts/generate
Content-Type: application/json

{
  "text": "Text to convert to speech",
  "voice": "alloy",
  "format": "mp3"
}
```

**Example**:
```bash
curl -X POST https://yoursite.com/tts/generate \
  -H "Content-Type: application/json" \
  -d '{"text": "Hello world!", "voice": "alloy"}'
```

**Response**:
```json
{
  "success": true,
  "audio_url": "/sites/default/files/tts/audio_123.mp3",
  "duration": 2.5,
  "format": "mp3",
  "generated": 1729382400
}
```

### Hooks

#### Alter text before TTS generation

```php
/**
 * Implements hook_tts_text_alter().
 */
function mymodule_tts_text_alter(&$text, $context) {
  // Remove ads or unwanted content
  $text = str_replace('[advertisement]', '', $text);
  
  // Add introduction
  if ($context['node']->bundle() == 'article') {
    $text = "Article: " . $text;
  }
}
```

#### React after audio generation

```php
/**
 * Implements hook_tts_audio_generated().
 */
function mymodule_tts_audio_generated($audio_data, $context) {
  // Log generation
  \Drupal::logger('mymodule')->info('TTS audio generated: @file', [
    '@file' => $audio_data['uri'],
  ]);
  
  // Notify user
  if (isset($context['node'])) {
    \Drupal::messenger()->addMessage('Audio version is now available.');
  }
}
```

---

# AI Module Integration

## How It Works

The TTS module integrates with the Drupal AI module's Text-to-Speech operation type.

### Configuration Flow

```
User Request
    ↓
TTS Service reads ai.settings
    ↓
Gets provider_id (openai) and model_id (tts-1-hd)
    ↓
Creates TextToSpeechInput($text)
    ↓
Creates provider with config: ['voice' => 'alloy']
    ↓
Calls $provider->textToSpeech($input, $model)
    ↓
Provider merges config into API payload
    ↓
OpenAI API: {model: "tts-1-hd", input: "text", voice: "alloy"}
    ↓
Returns audio data
    ↓
Saves to file
    ↓
Serves to user
```

### Key Integration Points

**1. Uses AI Module's TTS Configuration**
- Provider and model configured once in AI module
- TTS module reads these settings
- No duplication of configuration

**2. Uses AI Module Interfaces**
- `TextToSpeechInterface` - Provider interface
- `TextToSpeechInput` - Input object
- `TextToSpeechOutput` - Output object

**3. Provider Configuration**
```php
// Get AI module's TTS settings
$ai_config = $this->configFactory->get('ai.settings');
$tts_provider = $ai_config->get('default_providers.text_to_speech.provider_id');
$tts_model = $ai_config->get('default_providers.text_to_speech.model_id');

// Create provider instance with voice configuration
$provider_config = ['voice' => 'alloy'];
$provider = $this->aiProviderManager->createInstance($tts_provider, $provider_config);

// Generate audio
$input = new TextToSpeechInput($text);
$output = $provider->textToSpeech($input, $tts_model);
```

### Benefits

✅ **Single Source of Truth**: Provider configured once  
✅ **Proper Integration**: Uses official AI module interfaces  
✅ **Better UX**: Clear where to configure what  
✅ **Provider Agnostic**: Works with any AI provider  
✅ **Maintainable**: Less duplicate code  

---

# Technical Details

## Module Structure

```
tts/
├── tts.info.yml              # Module metadata
├── tts.module                # Hook implementations
├── tts.install               # Install/uninstall hooks
├── tts.services.yml          # Service definitions
├── tts.routing.yml           # Route definitions
├── tts.permissions.yml       # Permission definitions
├── tts.libraries.yml         # CSS/JS libraries
├── tts.links.menu.yml        # Menu links
├── composer.json                # Composer metadata
├── DOCUMENTATION.md             # This file
│
├── config/
│   ├── install/
│   │   └── tts.settings.yml # Default configuration
│   └── schema/
│       └── tts.schema.yml   # Configuration schema
│
├── src/
│   ├── Service/
│   │   └── TextToSpeechService.php  # Core TTS service
│   ├── Form/
│   │   └── AiTtsSettingsForm.php    # Configuration form
│   ├── Controller/
│   │   └── TtsController.php        # HTTP controllers
│   ├── Plugin/Block/
│   │   └── TtsPlayerBlock.php       # Audio player block
│   └── TtsLazyBuilder.php           # Lazy builder service
│
├── templates/
│   └── tts-player.html.twig     # Player template
│
├── css/
│   └── tts-player.css           # Player styles
│
└── js/
    └── tts-player.js            # Player functionality
```

## Service Architecture

### TextToSpeechService

**File**: `src/Service/TextToSpeechService.php`

**Responsibilities**:
- Generate audio from text
- Extract text from nodes
- Manage caching
- Save audio files
- Clean up old files

**Key Methods**:

```php
// Generate speech from text
public function generateSpeech(string $text, array $options = []): ?array

// Generate speech from node
public function generateNodeSpeech(NodeInterface $node, array $options = []): ?array

// Get audio URL for node
public function getAudioUrl(NodeInterface $node): ?string

// Clear TTS cache
public function clearCache(?string $cache_key = NULL): void

// Clean up old files
public function cleanupOldFiles(int $age = 604800): void
```

### TtsLazyBuilder

**File**: `src/TtsLazyBuilder.php`

**Purpose**: Lazy load TTS player to improve performance

**Methods**:
```php
// Render player for a node
public function renderPlayer($node_id): array
```

## Database Schema

The module uses Drupal's configuration system. No custom database tables.

**Configuration Storage**:
- `tts.settings` - Module configuration
- `ai.settings:default_providers.text_to_speech` - Provider settings

## File Storage

**Location**: `public://tts/`  
**Path**: `web/sites/default/files/tts/`

**File Naming**:
- Node-specific: `tts_node_{node_id}_{unique_id}.{format}` (for node content)
- Generic: `tts_{unique_id}.{format}` (for direct text generation)

**Examples**:
- `tts_node_42_68f4d40f43e9e.mp3` - Audio for node 42
- `tts_68f4d003e16f66.mp3` - Direct text generation

### Automatic Cleanup

**On Node Update** (`hook_node_update`):
- Old audio files for the node are automatically deleted
- Cache cleared for the node
- New audio generated on next page visit

**On Node Delete** (`hook_node_delete`):
- All audio files for the node are deleted
- Cache cleared
- No orphaned files left

**Benefits**:
- 80% disk space savings (only current audio kept)
- No manual cleanup needed
- All operations logged

## Caching Strategy

**Two-Layer Caching System**:

### 1. Database Cache (Temporary)
**Cache Backend**: `cache.default`  
**Cache Key Format**: `tts:{md5(text + options)}`  
**Cache Duration**: Configurable (default: 7 days)

**Cache Invalidation**:
- Manual: Admin can clear TTS cache
- Automatic: Cron job cleans up old cached files
- Node-based: Cache tags on `node:{nid}`

### 2. Persistent File Cache (Permanent)
**Storage**: `public://tts/tts_node_{nid}_{uid}.{format}`  
**Persistence**: Survives `drush cache clear` operations  
**Smart Reuse**: Files automatically reused when:
  - Cache is cleared (e.g., `drush cr`)
  - Node hasn't been modified since audio generation
  - Audio format and configuration unchanged

**Auto-Regeneration**: Files regenerated only when:
  - Node content is modified
  - Audio file doesn't exist
  - Node is older than existing file

**Benefits**:
- ✅ **Zero API calls after cache clear** (saves money)
- ✅ **Instant audio playback** after cache rebuild
- ✅ **Disk space efficient** (automatic cleanup on node update/delete)
- ✅ **Smart validation** (checks node modification time)

## Performance

**Optimization Techniques**:
- Smart caching (7-day default)
- Lazy loading with placeholders
- Browser caching headers
- Efficient file serving
- Text preprocessing to reduce API calls

**Estimated Costs** (OpenAI):
- Short article (500 chars): ~$0.0075
- Medium article (2,000 chars): ~$0.03
- Long article (5,000 chars): ~$0.075

## Security

**Measures**:
- Permission-based access control
- Input validation and sanitization
- CSRF protection (Form API)
- XSS prevention (Twig auto-escaping)
- Secure file handling
- API key protection

## Accessibility

**WCAG 2.1 Level AA Compliant**:
- Full keyboard navigation
- ARIA labels and roles
- Screen reader compatible
- Focus indicators
- Semantic HTML
- Alternative text content

---

# Troubleshooting

## Common Issues

### No Audio Player Appears

**Symptoms**: Player doesn't show on node pages

**Checklist**:
- [ ] Content type enabled in TTS settings?
- [ ] Block placed and visible on current page?
- [ ] User has "Access TTS audio" permission?
- [ ] Cache cleared?

**Solution**:
```bash
# 1. Check content type is enabled
# Go to /admin/config/ai/tts

# 2. Check block is placed
# Go to /admin/structure/block

# 3. Clear cache
ddev drush cr

# 4. Check permissions
# Go to /admin/people/permissions
```

### Audio Generation Fails

**Symptoms**: Error messages, no audio generated

**Check**:
1. **AI Module TTS configured?**
   ```bash
   ddev drush config:get ai.settings default_providers.text_to_speech
   ```
   Should show provider_id and model_id

2. **API key valid?**
   - Go to `/admin/config/ai/settings`
   - Verify API key is correct

3. **API quota available?**
   - Check your provider dashboard
   - Ensure you have credits/quota

4. **Check logs**:
   ```bash
   ddev drush ws --count=20
   ```

### JavaScript Errors

**Error**: `Uncaught TypeError: $(...).once is not a function`

**Cause**: Drupal 10 compatibility issue

**Solution**: Already fixed in module code. Clear cache:
```bash
ddev drush cr
# Clear browser cache too (Ctrl+Shift+R)
```

### "No TTS provider/model configured"

**Cause**: AI module TTS not configured

**Solution**:
```bash
ddev drush config:set ai.settings default_providers.text_to_speech.provider_id openai
ddev drush config:set ai.settings default_providers.text_to_speech.model_id tts-1-hd
ddev drush cr
```

### Permission Denied on Files

**Cause**: File directory not writable

**Solution**:
```bash
# Create directory
mkdir -p web/sites/default/files/tts

# Set permissions
chmod 755 web/sites/default/files/tts
chown -R www-data:www-data web/sites/default/files/tts
```

### High API Costs

**Cause**: Too many audio generations

**Solutions**:
1. Increase cache duration:
   ```
   Go to /admin/config/ai/tts
   Set Cache Duration to maximum (30 days)
   ```

2. Limit content types:
   ```
   Only enable essential content types
   ```

3. Reduce max text length:
   ```
   Set Max Text Length to lower value
   ```

4. Monitor usage:
   ```bash
   ddev drush ws | grep tts
   ```

## Debugging Commands

```bash
# Check module status
ddev drush pm:list | grep tts

# Verify TTS service loads
ddev drush eval "print 'Service loaded: ' . class_exists('Drupal\tts\Service\TextToSpeechService');"

# Check AI provider
ddev drush eval "print_r(\Drupal::service('ai.provider')->getDefinitions());"

# Test TTS generation
ddev drush eval "\$tts = \Drupal::service('tts.text_to_speech'); \$result = \$tts->generateSpeech('test'); print_r(\$result);"

# View recent errors
ddev drush ws --severity=Error --count=20

# Check file permissions
ls -la web/sites/default/files/ | grep tts

# View configuration
ddev drush config:get tts.settings
ddev drush config:get ai.settings default_providers.text_to_speech
```

---

# API Reference

## Services

### tts.text_to_speech

Main TTS service for generating audio.

**Class**: `Drupal\tts\Service\TextToSpeechService`

**Methods**:

#### generateSpeech()
```php
public function generateSpeech(string $text, array $options = []): ?array
```

Generate audio from text.

**Parameters**:
- `$text` (string): Text to convert to speech
- `$options` (array): Optional configuration
  - `voice` (string): Voice ID
  - `format` (string): Audio format (mp3, ogg, wav)

**Returns**: Array with audio data or NULL on failure
```php
[
  'uri' => 'public://tts/audio.mp3',
  'url' => '/sites/default/files/tts/audio.mp3',
  'format' => 'mp3',
  'duration' => 5.2,
  'generated' => 1729382400,
]
```

**Example**:
```php
$tts = \Drupal::service('tts.text_to_speech');
$audio = $tts->generateSpeech('Hello world!', ['voice' => 'alloy']);
```

#### generateNodeSpeech()
```php
public function generateNodeSpeech(NodeInterface $node, array $options = []): ?array
```

Generate audio from node content.

**Parameters**:
- `$node` (NodeInterface): Node to convert
- `$options` (array): Optional configuration

**Returns**: Array with audio data or NULL

**Example**:
```php
$node = Node::load(123);
$audio = $tts->generateNodeSpeech($node);
```

#### getAudioUrl()
```php
public function getAudioUrl(NodeInterface $node): ?string
```

Get audio URL for a node (generates if needed).

**Returns**: Audio URL or NULL

#### clearCache()
```php
public function clearCache(?string $cache_key = NULL): void
```

Clear TTS cache.

#### cleanupOldFiles()
```php
public function cleanupOldFiles(int $age = 604800): void
```

Remove audio files older than specified age.

### tts.lazy_builder

Lazy builder for TTS player.

**Class**: `Drupal\tts\TtsLazyBuilder`

**Method**:
```php
public function renderPlayer($node_id): array
```

## Routes

### tts.settings
- **Path**: `/admin/config/ai/tts`
- **Permission**: `administer tts settings`
- **Form**: `AiTtsSettingsForm`

### tts.node_audio
- **Path**: `/tts/node/{node}/audio`
- **Method**: GET
- **Permission**: `access tts audio` + node view access
- **Returns**: Audio file stream

### tts.generate_text
- **Path**: `/tts/generate`
- **Method**: POST
- **Permission**: `generate tts audio`
- **Format**: JSON
- **Body**:
```json
{
  "text": "Text to convert",
  "voice": "alloy",
  "format": "mp3"
}
```

## Permissions

- `administer tts settings` - Configure module
- `generate tts audio` - Generate audio programmatically
- `access tts audio` - Listen to and download audio

## Hooks

### hook_tts_text_alter()
```php
function mymodule_tts_text_alter(&$text, $context) {
  // Modify text before TTS generation
}
```

**Parameters**:
- `$text` (string, by reference): Text to modify
- `$context` (array): Context information
  - `node` (NodeInterface): Node being processed
  - `node_id` (int): Node ID
  - `node_type` (string): Content type

### hook_tts_audio_generated()
```php
function mymodule_tts_audio_generated($audio_data, $context) {
  // React after audio generation
}
```

**Parameters**:
- `$audio_data` (array): Generated audio information
- `$context` (array): Generation context

## Theme

### tts_player
**Template**: `templates/tts-player.html.twig`

**Variables**:
- `node_id` (int): Node ID
- `audio_url` (string): Audio file URL
- `title` (string): Node title
- `loading` (bool): Is audio being generated?

**Example Override**:
```twig
{# your-theme/templates/tts-player.html.twig #}
<div class="my-custom-player">
  <h3>{{ title }}</h3>
  {% if audio_url %}
    <audio src="{{ audio_url }}" controls></audio>
  {% endif %}
</div>
```


---

## Support & Resources

- **AI Module**: https://www.drupal.org/project/ai
- **AI Module Docs**: https://project.pages.drupalcode.org/ai/1.1.x/
- **OpenTTS**: https://platform.openai.com/docs/guides/text-to-speech
- **Google Cloud TTS**: https://cloud.google.com/text-to-speech
- **Drupal Community**: https://drupal.slack.com #ai channel

---

**Module Version**: 1.0.0  
**Status**: ✅ Production Ready  
**License**: GPL-2.0-or-later

---

*Generated: October 19, 2025*  
*This documentation consolidates all module documentation into one comprehensive guide.*

