# External Storage Configuration

The TTS module supports storing generated audio files on external storage services like CDN, AWS S3, Azure Blob Storage, and Google Cloud Storage. This offloads storage from your Drupal server and improves performance through CDN delivery.

## Supported Storage Types

1. **Local** (Default) - Drupal public files directory
2. **AWS S3** - Amazon S3 or S3-compatible services (DigitalOcean Spaces, Wasabi, etc.)
3. **Azure Blob Storage** - Microsoft Azure Blob Storage
4. **Google Cloud Storage** - Google Cloud Storage (GCS)
5. **Generic CDN** - Any CDN with an upload API

---

## Configuration

Navigate to: **Configuration → AI → Text-to-Speech** (`/admin/config/ai/tts`)

Open the **"External Storage"** section.

---

## AWS S3 Setup

### Prerequisites
- AWS account with S3 access
- S3 bucket created
- IAM user with appropriate permissions

### Required Permissions
```json
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": [
        "s3:PutObject",
        "s3:PutObjectAcl"
      ],
      "Resource": "arn:aws:s3:::your-bucket-name/*"
    }
  ]
}
```

### Configuration Steps

1. **Storage Type**: Select "AWS S3 or S3-compatible"
2. **S3 Bucket Name**: Enter your bucket name (e.g., `my-tts-audio`)
3. **AWS Region**: Enter region (e.g., `us-east-1`, `eu-west-1`)
4. **AWS Access Key ID**: Your IAM user access key
5. **AWS Secret Access Key**: Your IAM user secret key
6. **Custom S3 Endpoint** (Optional): For S3-compatible services
   - DigitalOcean Spaces: `https://nyc3.digitaloceanspaces.com`
   - Wasabi: `https://s3.wasabisys.com`
   - Backblaze B2: `https://s3.us-west-004.backblazeb2.com`
7. **Custom CDN Domain** (Optional): CloudFront or custom domain
   - Example: `https://cdn.example.com`

### S3-Compatible Services

**DigitalOcean Spaces:**
```yaml
Bucket: my-space-name
Region: nyc3
Endpoint: https://nyc3.digitaloceanspaces.com
Custom Domain: https://my-space-name.nyc3.cdn.digitaloceanspaces.com
```

**Wasabi:**
```yaml
Bucket: my-bucket
Region: us-east-1
Endpoint: https://s3.wasabisys.com
```

---

## Azure Blob Storage Setup

### Prerequisites
- Azure account
- Storage account created
- Container created with public access

### Configuration Steps

1. **Storage Type**: Select "Azure Blob Storage"
2. **Storage Account Name**: Your Azure storage account name
3. **Storage Account Key**: Primary or secondary access key
4. **Container Name**: Name of your blob container
5. **Custom CDN Domain** (Optional): Azure CDN endpoint
   - Example: `https://mycdn.azureedge.net`

### Finding Your Credentials

In Azure Portal:
1. Go to **Storage Accounts**
2. Select your storage account
3. Under **Security + networking** → **Access keys**
4. Copy the **Storage account name** and **Key**

---

## Google Cloud Storage Setup

### Prerequisites
- GCP account
- GCS bucket created with public access
- Service account with Storage Object Admin role

### Creating Service Account

1. Go to **IAM & Admin** → **Service Accounts**
2. Create a new service account
3. Grant role: **Storage Object Admin**
4. Create and download JSON key

### Configuration Steps

1. **Storage Type**: Select "Google Cloud Storage"
2. **GCS Bucket Name**: Your bucket name
3. **Project ID**: Your GCP project ID
4. **Service Account JSON**: Paste the entire JSON key content
5. **Custom CDN Domain** (Optional): Cloud CDN endpoint

### Bucket Permissions

Make bucket objects publicly readable:
```bash
gsutil iam ch allUsers:objectViewer gs://your-bucket-name
```

---

## Generic CDN Setup

For CDNs with custom upload APIs (e.g., BunnyCDN, KeyCDN, custom storage).

### Configuration Steps

1. **Storage Type**: Select "Generic CDN with API"
2. **CDN Base URL**: Where files will be accessible
   - Example: `https://cdn.example.com`
3. **Upload API Endpoint**: API endpoint for file uploads
   - Example: `https://storage.bunnycdn.com/my-storage-zone/`
4. **API Key**: Your CDN API key or access token

### Example: BunnyCDN

```yaml
CDN Base URL: https://my-zone.b-cdn.net
Upload API Endpoint: https://storage.bunnycdn.com/my-storage-zone/
API Key: your-bunnycdn-api-key
```

---

## How It Works

### Upload Process

1. Audio is generated using TTS provider
2. If external storage is configured:
   - File is uploaded to configured storage backend
   - Public URL is returned
   - If upload fails, falls back to local storage
3. If local storage (default):
   - File is saved to `public://tts/` directory

### URL Handling

- **External Storage**: Full CDN URL (e.g., `https://cdn.example.com/tts_node_123.mp3`)
- **Local Storage**: Drupal URI (e.g., `public://tts/tts_node_123.mp3`)

### Caching

- Existing audio files are reused when node content hasn't changed
- Cache clear operations don't regenerate audio files unnecessarily
- Old files are automatically deleted when nodes are updated/deleted

---

## Benefits of External Storage

### Performance
- **CDN Delivery**: Files served from edge locations closest to users
- **Reduced Server Load**: Audio delivery offloaded from Drupal
- **Faster Downloads**: Optimized CDN infrastructure

### Scalability
- **Unlimited Storage**: No local disk space concerns
- **Global Distribution**: Audio available worldwide
- **High Bandwidth**: CDN handles traffic spikes

### Cost
- **Reduced Hosting Costs**: Less storage on your server
- **Pay-as-you-go**: Only pay for storage/bandwidth used
- **CDN Efficiency**: Often cheaper than direct server delivery

---

## Troubleshooting

### Check Logs

View logs at: **Reports → Recent log messages** (filter by "tts")

### Common Issues

**S3 Upload Fails:**
- Verify IAM permissions
- Check bucket region matches configuration
- Ensure bucket allows public-read ACL
- Test credentials with AWS CLI

**Azure Upload Fails:**
- Verify account key is correct
- Check container exists and is public
- Ensure account name is lowercase

**GCS Upload Fails:**
- Verify service account has Storage Object Admin role
- Check JSON key is valid
- Ensure bucket has public access

**Generic CDN Fails:**
- Verify API endpoint is correct
- Check API key has upload permissions
- Test API with curl/Postman first

### Fallback Behavior

If external storage upload fails:
- Module automatically falls back to local storage
- Warning is logged in system logs
- Audio still works, just served locally

---

## Security Best Practices

### Credentials
- Store credentials securely
- Use IAM roles when possible (AWS)
- Rotate keys regularly
- Never commit credentials to code

### Access Control
- Use least-privilege IAM policies
- Enable bucket versioning for backups
- Monitor access logs
- Set up alerts for unusual activity

### Public Access
- Audio files must be publicly readable for playback
- Use signed URLs for private content (future feature)
- Consider IP allowlisting for upload APIs

---

## Cost Estimation

### AWS S3 (us-east-1)
- Storage: ~$0.023/GB/month
- Transfer out: $0.09/GB (first 10 TB)
- Requests: $0.005 per 1,000 PUT requests
- **Example**: 10,000 audio files (50 MB avg) = ~$12/month

### Azure Blob Storage
- Storage: ~$0.018/GB/month (Hot tier)
- Transfer out: $0.087/GB (first 5 GB free)
- Operations: $0.005 per 10,000 PUT operations
- **Example**: 10,000 audio files (50 MB avg) = ~$10/month

### Google Cloud Storage
- Storage: ~$0.020/GB/month
- Network egress: $0.12/GB (first 1 GB free)
- Operations: $0.05 per 10,000 PUT operations
- **Example**: 10,000 audio files (50 MB avg) = ~$11/month

### CDN Services
**BunnyCDN:**
- Storage: $0.01/GB/month
- Bandwidth: $0.01-0.045/GB (depends on region)
- **Example**: 10,000 files (50 MB avg) + 10 TB traffic = ~$6/month

---

## Module Dependencies

### PHP Extensions
- **OpenSSL**: Required for signing requests (all providers)
- **cURL**: Required for HTTP client (included in Drupal)
- **JSON**: Required for API communication (built-in)

### Composer Packages (Optional)
For enhanced support, install official SDKs:

```bash
# AWS SDK (optional, for advanced S3 features)
composer require aws/aws-sdk-php

# Azure SDK (optional, for advanced Blob features)
composer require microsoft/azure-storage-blob

# Google Cloud SDK (optional, for advanced GCS features)
composer require google/cloud-storage
```

*Note: The module works without SDKs using native PHP HTTP clients.*

---

## Future Enhancements

- Signed URLs for private audio content
- Multi-region redundancy
- Automatic failover between providers
- Background sync to multiple storage backends
- Storage usage statistics and monitoring
- File migration tool (local ↔ external)

---

## Support

For issues or questions:
- Check module logs: `/admin/reports/dblog`
- Review storage provider documentation
- Test credentials with provider's official tools
- Enable debug logging for detailed error messages

