# Text-to-Speech (TTS) Module

**Machine Name**: `tts`  
**Version**: 1.0.1  
**Drupal**: 10.x | 11.x

## Overview

The TTS module provides Text-to-Speech functionality for Drupal websites, converting text content into natural-sounding spoken audio using AI-powered voices. This module integrates with the [Drupal AI module](https://www.drupal.org/project/ai) to leverage various AI providers for generating high-quality speech synthesis.

## Features

### Core Features
- **Node Content Reading**: Automatically converts node page content into spoken audio
- **Natural AI Voices**: Uses AI-powered text-to-speech engines for natural-sounding voices
- **Multiple Content Types**: Support for any Drupal content type
- **Configurable Settings**: Admin interface to configure TTS providers and voice settings
- **REST API**: Programmatic access to generate TTS audio

### Player Customization 🎨
- **5 Player Styles**: Default, Compact, Minimal, Large, Widget (floating side player)
- **Widget Mode**: Circular floating player on the side of the page with gradient design
- **Toggleable Features**: Enable/disable download button, volume control individually
- **Custom Template Support**: Override with your own Twig template if needed

### Performance & Efficiency
- **Persistent File Caching**: Audio files survive cache clear operations, preventing unnecessary API calls 🎯
- **Smart Regeneration**: Files only regenerated when node content changes
- **Automatic Cleanup**: Old audio files deleted on node update/delete (80% disk space savings)
- **External Storage**: Support for CDN, AWS S3, Azure Blob, Google Cloud Storage for scalable delivery

### User Experience
- **Accessibility**: Improves content accessibility for visually impaired users
- **Mobile Responsive**: Beautiful design on all devices
- **Dark Mode Support**: Automatically adapts to user preferences
- **Keyboard Controls**: Full keyboard navigation support

## Requirements

- Drupal 10.x or 11.x
- [AI Module](https://www.drupal.org/project/ai) (1.1.x or later)
- PHP 8.1 or higher
- One or more AI provider modules configured (e.g., OpenAI, Google Cloud, Azure)

## Installation

### Using Composer (Recommended)

1. Install the AI module and its dependencies:
   ```bash
   composer require drupal/ai
   ```

2. Place this module in your Drupal installation:
   ```bash
   # If installing from a repository
   composer require drupal/tts
   
   # Or manually place in web/modules/custom/tts
   ```

3. Enable the module:
   ```bash
   drush en tts -y
   ```

### Manual Installation

1. Download and install the AI module from drupal.org
2. Download this module and place it in `/modules/custom/tts`
3. Enable the module via the Drupal admin interface: **Extend** > Enable "AI Text-to-Speech"

## Configuration

### 1. Configure AI Provider

Before using TTS, you must configure at least one AI provider:

1. Navigate to **Configuration** > **AI** > **Settings** (`/admin/config/ai/settings`)
2. Configure your preferred AI provider (OpenAI, Google Cloud TTS, Azure Cognitive Services, etc.)
3. Enter API credentials and test the connection

### 2. Configure TTS Settings

1. Navigate to **Configuration** > **AI** > **Text-to-Speech Settings** (`/admin/config/ai/tts`)
2. Configure the following options:

   - **AI Provider**: Select which AI provider to use for TTS
   - **Voice Model**: Choose the voice model/persona
   - **Voice Language**: Select language and accent
   - **Speaking Rate**: Adjust speech speed (0.5x to 2.0x)
   - **Audio Format**: Choose output format (MP3, OGG, WAV)
   - **Content Types**: Select which content types should have TTS enabled
   - **Field Selection**: Choose which fields to include in TTS (body, custom fields)
   - **Cache Duration**: Set how long to cache generated audio files

### 3. Enable TTS Block

1. Navigate to **Structure** > **Block layout**
2. Place the "TTS Audio Player" block in your desired region
3. Configure block visibility settings (e.g., show only on node pages)

### 4. Customize Player (Optional) 🎨

Customize the audio player appearance and features:

1. Navigate to **Configuration** > **AI** > **Text-to-Speech Settings** (`/admin/config/ai/tts`)
2. Expand the **"Player Customization"** section
3. Configure player options:

#### Player Style
Choose from 4 visual styles:
- **Default**: Full-featured player with all controls (inline)
- **Compact**: Smaller player for tight spaces (inline)
- **Minimal**: Bare-bones player with essential controls only (inline)
- **Large**: Enhanced visibility for accessibility (inline)
- **Widget**: Circular floating player on the side of the page with beautiful gradient design 🆕

#### Control Options
- **Show Volume Control**: Display volume slider and mute button
- **Show Download Button**: Allow users to download audio files

#### Advanced
- **Custom Template**: Override the default player with your own Twig template
  - Example: `@mytheme/custom-audio-player.html.twig`
  - Allows complete control over player HTML and styling

### External Storage 💾

Store audio files on external storage for better performance and scalability:

#### Supported Storage Types
- **Local** (Default): Drupal public files directory
- **AWS S3**: Amazon S3 or S3-compatible (DigitalOcean Spaces, Wasabi, Backblaze B2)
- **Azure Blob Storage**: Microsoft Azure Blob Storage
- **Google Cloud Storage**: Google Cloud Storage (GCS)
- **Generic CDN**: Any CDN with upload API (BunnyCDN, KeyCDN, custom)

#### Benefits
- **CDN Delivery**: Files served from edge locations worldwide
- **Reduced Server Load**: Audio delivery offloaded from Drupal
- **Unlimited Scalability**: No local disk space constraints
- **Cost Effective**: Pay-as-you-go pricing, often cheaper than direct hosting
- **Automatic Fallback**: Falls back to local storage if external upload fails

#### Configuration
Navigate to: **Configuration → AI → Text-to-Speech** (`/admin/config/ai/tts`)

Open the **"External Storage"** section and configure your preferred provider:
- Enter bucket/container name and region
- Add access credentials (access keys, account keys, service account JSON)
- (Optional) Add custom CDN domain for branded delivery
- Save configuration - uploads happen automatically

For detailed setup guides for each provider, see **DOCUMENTATION.md**.

## Usage

### For Content Viewers

Once configured, users will see an audio player widget on enabled node pages:

1. Visit any node page with TTS enabled
2. Click the "Listen to this page" button or play icon
3. Use standard audio controls (play, pause, volume, playback speed)

### For Site Builders

#### Programmatic Usage

Generate TTS audio programmatically:

```php
// Get the TTS service
$ttsService = \Drupal::service('tts.text_to_speech');

// Generate audio from text
$text = "Hello, this is a test of the text-to-speech system.";
$audioData = $ttsService->generateSpeech($text);

// Generate audio from a node
$node = \Drupal\node\Entity\Node::load(1);
$nodeAudioData = $ttsService->generateNodeSpeech($node);

// Get audio URL
$audioUrl = $ttsService->getAudioUrl($node);
```

#### Custom Integration

Integrate TTS into custom templates:

```php
// In your template or preprocess function
$tts_service = \Drupal::service('tts.text_to_speech');
$variables['audio_url'] = $tts_service->getAudioUrl($node);
```

```twig
{# In your Twig template #}
{% if audio_url %}
  <audio controls>
    <source src="{{ audio_url }}" type="audio/mpeg">
    Your browser does not support the audio element.
  </audio>
{% endif %}
```

### REST API Endpoints

#### Generate TTS for Node
```
GET /tts/node/{node_id}/audio
```

**Response**: Audio file stream

#### Generate TTS from Text
```
POST /tts/generate
Content-Type: application/json

{
  "text": "Text to convert to speech",
  "language": "en-US",
  "voice": "neural"
}
```

**Response**:
```json
{
  "audio_url": "/sites/default/files/tts/audio_123.mp3",
  "duration": 5.2,
  "format": "mp3"
}
```

## Supported AI Providers

The module supports any TTS provider configured through the AI module:

- **OpenTTS**: High-quality voices with natural intonation
- **Google Cloud Text-to-Speech**: Multiple languages and WaveNet voices
- **Azure Cognitive Services**: Neural voices with emotion control
- **Amazon Polly**: Wide language support with neural voices
- **IBM Watson**: Enterprise-grade TTS

## Performance & Caching

- Generated audio files are cached to improve performance
- Cache duration is configurable (default: 7 days)
- Audio files are stored in `public://tts/` directory
- Large content is automatically chunked for processing

## Accessibility

This module enhances website accessibility by:

- Providing audio alternatives for text content
- Supporting screen reader users with ARIA labels
- Keyboard navigation support for audio player
- Compliant with WCAG 2.1 Level AA standards

## Permissions

The module provides the following permissions:

- **Administer TTS settings**: Configure module settings
- **Generate TTS audio**: Generate audio from text (for custom implementations)
- **Access TTS audio**: View and play TTS audio on nodes

Configure permissions at: **People** > **Permissions**

## Troubleshooting

### No audio player appears

1. Check that the content type is enabled in TTS settings
2. Verify the TTS block is placed and visible
3. Clear Drupal cache: `drush cr`

### Audio generation fails

1. Verify AI provider credentials are correct
2. Check API quota/rate limits on your AI provider account
3. Review Drupal logs: **Reports** > **Recent log messages**
4. Test AI provider connection in AI module settings

### Poor audio quality

1. Try different voice models in TTS settings
2. Adjust speaking rate and pitch settings
3. Consider using a different AI provider
4. Ensure text content is properly formatted (remove HTML artifacts)

### Cache issues

1. Clear TTS cache: `drush cache:clear tts`
2. Rebuild cache: `drush cr`
3. Check file permissions on `public://tts/` directory

## Development

### Hooks

The module provides hooks for custom integration:

```php
/**
 * Alter text before TTS generation.
 */
function hook_tts_text_alter(&$text, $context) {
  // Modify text before speech generation
  $text = str_replace('[custom_token]', 'replacement', $text);
}

/**
 * Alter TTS configuration.
 */
function hook_tts_config_alter(&$config, $context) {
  // Modify TTS configuration per request
  if ($context['node']->bundle() == 'article') {
    $config['voice'] = 'professional';
  }
}

/**
 * React after audio generation.
 */
function hook_tts_audio_generated($audio_data, $context) {
  // Perform actions after audio is generated
  \Drupal::logger('my_module')->info('TTS audio generated: @file', [
    '@file' => $audio_data['uri'],
  ]);
}
```


## Credits

- **Maintainer**: [Mustapha ghazali](https://www.drupal.org/u/gmustapha) 
- **Site**: [ghazali.cc](https://ghazali.cc/) 
- **Built with**: [Drupal AI Module](https://www.drupal.org/project/ai)
- **Documentation**: [AI Module Documentation](https://project.pages.drupalcode.org/ai/1.1.x/)

## License

This module is licensed under the GNU General Public License v2.0 or later.

## Support

- **Issue Queue**: [Report bugs and feature requests](https://www.drupal.org/project/issues/tts)
- **Documentation**: [Module documentation](https://www.drupal.org/docs/contributed-modules/tts)
- **Community**: [Drupal Slack #ai channel](https://drupal.slack.com)

## Related Modules

- [AI](https://www.drupal.org/project/ai) - Base AI framework
- [AI Automator](https://www.drupal.org/project/ai_automator) - AI workflow automation
- [Accessibility](https://www.drupal.org/project/accessibility) - Additional accessibility tools

---