/**
 * @file
 * JavaScript for the TTS audio player.
 */

(function ($, Drupal, once) {
  'use strict';

  /**
   * Initialize the TTS audio player.
   */
  Drupal.behaviors.ttsPlayer = {
    attach: function (context, settings) {
      // Use Drupal 10+ once() utility
      once('tts-player', '.tts-player', context).forEach(function (element) {
        const $player = $(element);
        const $audio = $player.find('.tts-player__audio');

        if ($audio.length === 0) {
          return;
        }

        const audio = $audio[0];
        const $playButton = $player.find('.tts-player__button--play');
        const $pauseButton = $player.find('.tts-player__button--pause');
        const $progressBar = $player.find('.tts-player__progress-bar');
        const $progressFill = $player.find('.tts-player__progress-fill');
        const $timeCurrent = $player.find('.tts-player__time-current');
        const $timeDuration = $player.find('.tts-player__time-duration');
        const $volumeSlider = $player.find('.tts-player__volume-slider');
        const $volumeButton = $player.find('.tts-player__button--volume');

        // Format time helper.
        function formatTime(seconds) {
          if (isNaN(seconds)) {
            return '0:00';
          }
          const minutes = Math.floor(seconds / 60);
          const secs = Math.floor(seconds % 60);
          return minutes + ':' + (secs < 10 ? '0' : '') + secs;
        }

        // Play button handler.
        $playButton.on('click', function () {
          audio.play();
          $playButton.hide();
          $pauseButton.show();
        });

        // Pause button handler.
        $pauseButton.on('click', function () {
          audio.pause();
          $pauseButton.hide();
          $playButton.show();
        });

        // Update progress and time.
        audio.addEventListener('timeupdate', function () {
          const progress = (audio.currentTime / audio.duration) * 100;
          $progressFill.css('width', progress + '%');
          $progressBar.attr('aria-valuenow', progress);
          $timeCurrent.text(formatTime(audio.currentTime));
        });

        // Update duration when metadata is loaded.
        audio.addEventListener('loadedmetadata', function () {
          $timeDuration.text(formatTime(audio.duration));
        });

        // Handle audio end.
        audio.addEventListener('ended', function () {
          $pauseButton.hide();
          $playButton.show();
          $progressFill.css('width', '0%');
          audio.currentTime = 0;
        });

        // Seek functionality.
        $progressBar.on('click', function (e) {
          const rect = this.getBoundingClientRect();
          const percent = (e.clientX - rect.left) / rect.width;
          audio.currentTime = percent * audio.duration;
        });

        // Volume control.
        $volumeSlider.on('input', function () {
          audio.volume = this.value / 100;
        });

        // Volume button toggle mute.
        $volumeButton.on('click', function () {
          if (audio.volume > 0) {
            audio.dataset.previousVolume = audio.volume;
            audio.volume = 0;
            $volumeSlider.val(0);
          } else {
            const previousVolume = parseFloat(audio.dataset.previousVolume) || 1;
            audio.volume = previousVolume;
            $volumeSlider.val(previousVolume * 100);
          }
        });

        // Keyboard controls.
        $player.on('keydown', function (e) {
          switch (e.key) {
            case ' ':
            case 'k':
              e.preventDefault();
              if (audio.paused) {
                $playButton.click();
              } else {
                $pauseButton.click();
              }
              break;

            case 'ArrowLeft':
              e.preventDefault();
              audio.currentTime = Math.max(0, audio.currentTime - 5);
              break;

            case 'ArrowRight':
              e.preventDefault();
              audio.currentTime = Math.min(audio.duration, audio.currentTime + 5);
              break;

            case 'ArrowUp':
              e.preventDefault();
              audio.volume = Math.min(1, audio.volume + 0.1);
              $volumeSlider.val(audio.volume * 100);
              break;

            case 'ArrowDown':
              e.preventDefault();
              audio.volume = Math.max(0, audio.volume - 0.1);
              $volumeSlider.val(audio.volume * 100);
              break;

            case 'm':
              e.preventDefault();
              $volumeButton.click();
              break;
          }
        });

        // Handle errors.
        audio.addEventListener('error', function (e) {
          console.error('Audio error:', e);
          $player.find('.tts-player__container').html(
            '<div class="tts-player__error">' +
            Drupal.t('Unable to load audio. Please try again later.') +
            '</div>'
          );
        });

        // AJAX generation for loading state.
        if ($player.hasClass('tts-player--loading')) {
          const nodeId = $player.data('node-id');
          if (nodeId) {
            $.ajax({
              url: '/tts/node/' + nodeId + '/audio',
              method: 'GET',
              success: function (data) {
                // Reload the page or update the player with new audio.
                location.reload();
              },
              error: function () {
                $player.find('.tts-player__loading').html(
                  '<div class="tts-player__error">' +
                  Drupal.t('Failed to generate audio.') +
                  '</div>'
                );
              }
            });
          }
        }

        // Analytics tracking (optional).
        if (typeof gtag !== 'undefined') {
          $playButton.on('click', function () {
            gtag('event', 'play', {
              'event_category': 'TTS',
              'event_label': 'Node ' + $player.data('node-id')
            });
          });
        }
      });
    }
  };

})(jQuery, Drupal, once);
