<?php

namespace Drupal\tts\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\File\FileSystemInterface;
use Drupal\node\NodeInterface;
use Drupal\tts\Service\TextToSpeechService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;

/**
 * Controller for TTS operations.
 */
class TtsController extends ControllerBase {

  /**
   * The TTS service.
   *
   * @var \Drupal\tts\Service\TextToSpeechService
   */
  protected $ttsService;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * Constructs a TtsController object.
   *
   * @param \Drupal\tts\Service\TextToSpeechService $tts_service
   *   The TTS service.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   */
  public function __construct(
    TextToSpeechService $tts_service,
    FileSystemInterface $file_system,
  ) {
    $this->ttsService = $tts_service;
    $this->fileSystem = $file_system;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('tts.text_to_speech'),
      $container->get('file_system')
    );
  }

  /**
   * Generate and serve audio for a node.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node to convert to audio.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The audio file response.
   */
  public function generateNodeAudio(NodeInterface $node): Response {
    // Check access.
    if (!$node->access('view')) {
      throw new NotFoundHttpException();
    }

    // Generate audio.
    $audio_data = $this->ttsService->generateNodeSpeech($node);

    if (!$audio_data) {
      return new JsonResponse([
        'error' => $this->t('Unable to generate audio for this content.'),
      ], Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    // Check if URI is an external URL (external storage).
    if ($this->isExternalUrl($audio_data['uri'])) {
      // Return JSON with external URL.
      return new JsonResponse([
        'success' => TRUE,
        'audio_url' => $audio_data['uri'],
        'format' => $audio_data['format'],
        'duration' => $audio_data['duration'] ?? NULL,
        'generated' => $audio_data['generated'] ?? time(),
      ]);
    }

    // Return audio file for local storage.
    return $this->serveAudioFile($audio_data['uri'], $audio_data['format']);
  }

  /**
   * Generate audio from submitted text.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with audio URL.
   */
  public function generateFromText(Request $request): JsonResponse {
    // Get request data.
    $data = json_decode($request->getContent(), TRUE);

    if (!isset($data['text']) || empty($data['text'])) {
      throw new BadRequestHttpException('Text parameter is required.');
    }

    $text = $data['text'];
    $options = [];

    // Optional parameters.
    if (isset($data['language'])) {
      $options['language'] = $data['language'];
    }
    if (isset($data['voice'])) {
      $options['voice'] = $data['voice'];
    }
    if (isset($data['rate'])) {
      $options['rate'] = (float) $data['rate'];
    }
    if (isset($data['format'])) {
      $options['format'] = $data['format'];
    }

    // Generate audio.
    $audio_data = $this->ttsService->generateSpeech($text, $options);

    if (!$audio_data) {
      return new JsonResponse([
        'error' => $this->t('Unable to generate audio.'),
      ], Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    // Return audio information.
    return new JsonResponse([
      'success' => TRUE,
      'audio_url' => $audio_data['url'],
      'duration' => $audio_data['duration'],
      'format' => $audio_data['format'],
      'generated' => $audio_data['generated'],
    ]);
  }

  /**
   * Serve a cached audio file.
   *
   * @param string $hash
   *   The file hash.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The audio file response.
   */
  public function serveCachedAudio(string $hash): Response {
    // Sanitize hash.
    $hash = preg_replace('/[^a-zA-Z0-9_-]/', '', $hash);

    // Try to find the file.
    $directory = 'public://tts';
    $possible_extensions = ['mp3', 'ogg', 'wav'];

    foreach ($possible_extensions as $ext) {
      $uri = $directory . '/' . $hash . '.' . $ext;
      if (file_exists($uri)) {
        return $this->serveAudioFile($uri, $ext);
      }
    }

    throw new NotFoundHttpException('Audio file not found.');
  }

  /**
   * Serve an audio file.
   *
   * @param string $uri
   *   The file URI.
   * @param string $format
   *   The audio format.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The file response.
   */
  protected function serveAudioFile(string $uri, string $format): Response {
    $path = $this->fileSystem->realpath($uri);

    if (!file_exists($path)) {
      throw new NotFoundHttpException('Audio file not found.');
    }

    // Determine MIME type.
    $mime_types = [
      'mp3' => 'audio/mpeg',
      'ogg' => 'audio/ogg',
      'wav' => 'audio/wav',
    ];
    $mime_type = $mime_types[$format] ?? 'application/octet-stream';

    // Create response.
    $response = new BinaryFileResponse($path);
    $response->headers->set('Content-Type', $mime_type);
    $response->headers->set('Content-Disposition', 'inline; filename="tts_audio.' . $format . '"');
    $response->headers->set('Cache-Control', 'public, max-age=604800');

    return $response;
  }

  /**
   * Check if a URI is an external URL.
   *
   * @param string $uri
   *   The URI to check.
   *
   * @return bool
   *   TRUE if external URL, FALSE otherwise.
   */
  protected function isExternalUrl(string $uri): bool {
    return (bool) preg_match('/^https?:\/\//', $uri);
  }

}
