<?php

namespace Drupal\tts\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure TTS settings.
 */
class TtsSettingsForm extends ConfigFormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a TtsSettingsForm object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
  ) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['tts.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'tts_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('tts.settings');

    $form['provider_info'] = [
      '#type' => 'item',
      '#title' => $this->t('AI Provider & Model'),
      '#markup' => $this->t('The TTS provider and model are configured in the <a href="@url">AI module settings</a> under "Text To Speech" section.', [
        '@url' => '/admin/config/ai/settings',
      ]),
    ];

    $form['voice_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Voice Settings'),
      '#open' => TRUE,
    ];

    $form['voice_settings']['voice_model'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Voice Model'),
      '#description' => $this->t('The voice model to use (e.g., "alloy", "nova", "shimmer" for OpenAI). Check your provider documentation for available voices.'),
      '#default_value' => $config->get('voice_model'),
      '#required' => TRUE,
    ];

    $form['output_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Output Settings'),
      '#open' => TRUE,
    ];

    $form['output_settings']['audio_format'] = [
      '#type' => 'select',
      '#title' => $this->t('Audio Format'),
      '#description' => $this->t('The output audio format. MP3 is recommended for best compatibility.'),
      '#options' => [
        'mp3' => $this->t('MP3'),
        'ogg' => $this->t('OGG Vorbis'),
        'wav' => $this->t('WAV'),
      ],
      '#default_value' => $config->get('audio_format'),
      '#required' => TRUE,
    ];

    $form['output_settings']['max_text_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum Text Length'),
      '#description' => $this->t('Maximum number of characters to process. Text longer than this will be truncated.'),
      '#default_value' => $config->get('max_text_length'),
      '#min' => 100,
      '#max' => 100000,
      '#required' => TRUE,
    ];

    $form['content_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Content Settings'),
      '#open' => TRUE,
    ];

    // Get available content types.
    $content_types = $this->getContentTypes();

    $form['content_settings']['enabled_content_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Enabled Content Types'),
      '#description' => $this->t('Select which content types should have TTS enabled.'),
      '#options' => $content_types,
      '#default_value' => $config->get('enabled_content_types') ?? [],
    ];

    $form['content_settings']['included_fields'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Included Fields'),
      '#description' => $this->t('Enter field machine names to include in TTS, one per line (e.g., body, field_description). The title is always included.'),
      '#default_value' => implode("\n", $config->get('included_fields') ?? ['body']),
      '#rows' => 5,
    ];

    $form['content_settings']['auto_generate'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto-generate on node save'),
      '#description' => $this->t('Automatically generate TTS audio when a node is saved. Warning: This may increase save times and API costs.'),
      '#default_value' => $config->get('auto_generate'),
    ];

    $form['cache_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Cache Settings'),
      '#open' => FALSE,
    ];

    $form['cache_settings']['cache_duration'] = [
      '#type' => 'number',
      '#title' => $this->t('Cache Duration'),
      '#description' => $this->t('How long to cache generated audio files (in seconds). Default is 604800 (7 days).'),
      '#default_value' => $config->get('cache_duration'),
      '#min' => 3600,
      '#max' => 2592000,
      '#required' => TRUE,
    ];

    $form['display_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Display Settings'),
      '#open' => FALSE,
    ];

    $form['display_settings']['show_player_block'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show player block'),
      '#description' => $this->t('Display the audio player block on enabled content types.'),
      '#default_value' => $config->get('show_player_block'),
    ];

    $form['display_settings']['player_position'] = [
      '#type' => 'select',
      '#title' => $this->t('Player Position'),
      '#description' => $this->t('Where to display the audio player on the page.'),
      '#options' => [
        'top' => $this->t('Top of content'),
        'bottom' => $this->t('Bottom of content'),
      ],
      '#default_value' => $config->get('player_position'),
      '#states' => [
        'visible' => [
          ':input[name="show_player_block"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Player Customization Settings.
    $form['player_customization'] = [
      '#type' => 'details',
      '#title' => $this->t('Player Customization'),
      '#description' => $this->t('Customize the appearance and functionality of the audio player.'),
      '#open' => FALSE,
    ];

    $form['player_customization']['player_style'] = [
      '#type' => 'select',
      '#title' => $this->t('Player Style'),
      '#description' => $this->t('Choose the visual style of the audio player.'),
      '#options' => [
        'default' => $this->t('Default (Full features)'),
        'compact' => $this->t('Compact (Small size)'),
        'minimal' => $this->t('Minimal (Bare controls)'),
        'large' => $this->t('Large (Enhanced visibility)'),
        'widget' => $this->t('Widget (Circular side player)'),
      ],
      '#default_value' => $config->get('player_style') ?: 'default',
    ];

    $form['player_customization']['show_volume_control'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show volume control'),
      '#description' => $this->t('Display volume slider and mute button.'),
      '#default_value' => $config->get('show_volume_control') ?? TRUE,
    ];

    $form['player_customization']['show_download_button'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show download button'),
      '#description' => $this->t('Allow users to download the audio file.'),
      '#default_value' => $config->get('show_download_button') ?? TRUE,
    ];

    $form['player_customization']['custom_template'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom template (advanced)'),
      '#description' => $this->t('Override the default player template with a custom Twig template. Example: @mytheme/custom-player.html.twig. Leave empty to use the default.'),
      '#default_value' => $config->get('custom_template') ?? '',
      '#maxlength' => 255,
    ];

    // External Storage Settings.
    $form['storage'] = [
      '#type' => 'details',
      '#title' => $this->t('External Storage'),
      '#description' => $this->t('Configure external storage for audio files (CDN, AWS S3, Azure Blob, Google Cloud Storage).'),
      '#open' => FALSE,
    ];

    $storage_settings = $config->get('storage_settings') ?? [];

    $form['storage']['storage_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Storage Type'),
      '#description' => $this->t('Choose where to store generated audio files.'),
      '#options' => [
        'local' => $this->t('Local (Drupal public files)'),
        's3' => $this->t('AWS S3 or S3-compatible'),
        'azure' => $this->t('Azure Blob Storage'),
        'gcs' => $this->t('Google Cloud Storage'),
        'cdn' => $this->t('Generic CDN with API'),
      ],
      '#default_value' => $config->get('storage_type') ?: 'local',
    ];

    // AWS S3 Settings.
    $form['storage']['s3'] = [
      '#type' => 'details',
      '#title' => $this->t('AWS S3 Settings'),
      '#open' => FALSE,
      '#states' => [
        'visible' => [
          ':input[name="storage_type"]' => ['value' => 's3'],
        ],
      ],
    ];

    $form['storage']['s3']['s3_bucket'] = [
      '#type' => 'textfield',
      '#title' => $this->t('S3 Bucket Name'),
      '#default_value' => $storage_settings['s3']['bucket'] ?? '',
      '#description' => $this->t('The name of your S3 bucket.'),
    ];

    $form['storage']['s3']['s3_region'] = [
      '#type' => 'textfield',
      '#title' => $this->t('AWS Region'),
      '#default_value' => $storage_settings['s3']['region'] ?? 'us-east-1',
      '#description' => $this->t('AWS region (e.g., us-east-1, eu-west-1).'),
    ];

    $form['storage']['s3']['s3_access_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('AWS Access Key ID'),
      '#default_value' => $storage_settings['s3']['access_key'] ?? '',
    ];

    $form['storage']['s3']['s3_secret_key'] = [
      '#type' => 'password',
      '#title' => $this->t('AWS Secret Access Key'),
      '#description' => $this->t('Leave empty to keep current value.'),
    ];

    $form['storage']['s3']['s3_endpoint'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom S3 Endpoint (optional)'),
      '#default_value' => $storage_settings['s3']['endpoint'] ?? '',
      '#description' => $this->t('For S3-compatible services like DigitalOcean Spaces, Wasabi, etc.'),
    ];

    $form['storage']['s3']['s3_custom_domain'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom CDN Domain (optional)'),
      '#default_value' => $storage_settings['s3']['custom_domain'] ?? '',
      '#description' => $this->t('If using CloudFront or custom CDN domain (e.g., https://cdn.example.com).'),
    ];

    // Azure Blob Storage Settings.
    $form['storage']['azure'] = [
      '#type' => 'details',
      '#title' => $this->t('Azure Blob Storage Settings'),
      '#open' => FALSE,
      '#states' => [
        'visible' => [
          ':input[name="storage_type"]' => ['value' => 'azure'],
        ],
      ],
    ];

    $form['storage']['azure']['azure_account_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Storage Account Name'),
      '#default_value' => $storage_settings['azure']['account_name'] ?? '',
    ];

    $form['storage']['azure']['azure_account_key'] = [
      '#type' => 'password',
      '#title' => $this->t('Storage Account Key'),
      '#description' => $this->t('Leave empty to keep current value.'),
    ];

    $form['storage']['azure']['azure_container'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Container Name'),
      '#default_value' => $storage_settings['azure']['container'] ?? '',
    ];

    $form['storage']['azure']['azure_custom_domain'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom CDN Domain (optional)'),
      '#default_value' => $storage_settings['azure']['custom_domain'] ?? '',
      '#description' => $this->t('If using Azure CDN or custom domain.'),
    ];

    // Google Cloud Storage Settings.
    $form['storage']['gcs'] = [
      '#type' => 'details',
      '#title' => $this->t('Google Cloud Storage Settings'),
      '#open' => FALSE,
      '#states' => [
        'visible' => [
          ':input[name="storage_type"]' => ['value' => 'gcs'],
        ],
      ],
    ];

    $form['storage']['gcs']['gcs_bucket'] = [
      '#type' => 'textfield',
      '#title' => $this->t('GCS Bucket Name'),
      '#default_value' => $storage_settings['gcs']['bucket'] ?? '',
    ];

    $form['storage']['gcs']['gcs_project_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Project ID'),
      '#default_value' => $storage_settings['gcs']['project_id'] ?? '',
    ];

    $form['storage']['gcs']['gcs_credentials_json'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Service Account JSON'),
      '#default_value' => $storage_settings['gcs']['credentials_json'] ?? '',
      '#description' => $this->t('Paste the entire JSON key file content from your service account.'),
      '#rows' => 5,
    ];

    $form['storage']['gcs']['gcs_custom_domain'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom CDN Domain (optional)'),
      '#default_value' => $storage_settings['gcs']['custom_domain'] ?? '',
      '#description' => $this->t('If using Cloud CDN or custom domain.'),
    ];

    // Generic CDN Settings.
    $form['storage']['cdn'] = [
      '#type' => 'details',
      '#title' => $this->t('Generic CDN Settings'),
      '#open' => FALSE,
      '#states' => [
        'visible' => [
          ':input[name="storage_type"]' => ['value' => 'cdn'],
        ],
      ],
    ];

    $form['storage']['cdn']['cdn_base_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('CDN Base URL'),
      '#default_value' => $storage_settings['cdn']['base_url'] ?? '',
      '#description' => $this->t('The base URL where files will be accessible (e.g., https://cdn.example.com).'),
    ];

    $form['storage']['cdn']['cdn_api_endpoint'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Upload API Endpoint'),
      '#default_value' => $storage_settings['cdn']['api_endpoint'] ?? '',
      '#description' => $this->t('The API endpoint for uploading files.'),
    ];

    $form['storage']['cdn']['cdn_api_key'] = [
      '#type' => 'password',
      '#title' => $this->t('API Key'),
      '#description' => $this->t('Leave empty to keep current value.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // Validate included fields format.
    $fields_text = $form_state->getValue('included_fields');
    $fields = array_filter(array_map('trim', explode("\n", $fields_text)));

    foreach ($fields as $field) {
      if (!preg_match('/^[a-z_][a-z0-9_]*$/', $field)) {
        $form_state->setErrorByName('included_fields', $this->t('Invalid field name: @field. Field names must contain only lowercase letters, numbers, and underscores.', ['@field' => $field]));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Process included fields.
    $fields_text = $form_state->getValue('included_fields');
    $fields = array_filter(array_map('trim', explode("\n", $fields_text)));

    // Process enabled content types.
    $enabled_types = array_filter($form_state->getValue('enabled_content_types'));

    $config = $this->config('tts.settings');
    $storage_settings = $config->get('storage_settings') ?? [];

    // Update storage settings, preserving passwords if empty.
    $storage_type = $form_state->getValue('storage_type');

    // S3 settings.
    $storage_settings['s3']['bucket'] = $form_state->getValue('s3_bucket') ?? '';
    $storage_settings['s3']['region'] = $form_state->getValue('s3_region') ?? 'us-east-1';
    $storage_settings['s3']['access_key'] = $form_state->getValue('s3_access_key') ?? '';
    if (!empty($form_state->getValue('s3_secret_key'))) {
      $storage_settings['s3']['secret_key'] = $form_state->getValue('s3_secret_key');
    }
    $storage_settings['s3']['endpoint'] = $form_state->getValue('s3_endpoint') ?? '';
    $storage_settings['s3']['custom_domain'] = $form_state->getValue('s3_custom_domain') ?? '';

    // Azure settings.
    $storage_settings['azure']['account_name'] = $form_state->getValue('azure_account_name') ?? '';
    if (!empty($form_state->getValue('azure_account_key'))) {
      $storage_settings['azure']['account_key'] = $form_state->getValue('azure_account_key');
    }
    $storage_settings['azure']['container'] = $form_state->getValue('azure_container') ?? '';
    $storage_settings['azure']['custom_domain'] = $form_state->getValue('azure_custom_domain') ?? '';

    // GCS settings.
    $storage_settings['gcs']['bucket'] = $form_state->getValue('gcs_bucket') ?? '';
    $storage_settings['gcs']['project_id'] = $form_state->getValue('gcs_project_id') ?? '';
    $storage_settings['gcs']['credentials_json'] = $form_state->getValue('gcs_credentials_json') ?? '';
    $storage_settings['gcs']['custom_domain'] = $form_state->getValue('gcs_custom_domain') ?? '';

    // CDN settings.
    $storage_settings['cdn']['base_url'] = $form_state->getValue('cdn_base_url') ?? '';
    $storage_settings['cdn']['api_endpoint'] = $form_state->getValue('cdn_api_endpoint') ?? '';
    if (!empty($form_state->getValue('cdn_api_key'))) {
      $storage_settings['cdn']['api_key'] = $form_state->getValue('cdn_api_key');
    }

    $config
      ->set('voice_model', $form_state->getValue('voice_model'))
      ->set('audio_format', $form_state->getValue('audio_format'))
      ->set('enabled_content_types', array_values($enabled_types))
      ->set('included_fields', array_values($fields))
      ->set('cache_duration', $form_state->getValue('cache_duration'))
      ->set('max_text_length', $form_state->getValue('max_text_length'))
      ->set('auto_generate', $form_state->getValue('auto_generate'))
      ->set('show_player_block', $form_state->getValue('show_player_block'))
      ->set('player_position', $form_state->getValue('player_position'))
      ->set('player_style', $form_state->getValue('player_style'))
      ->set('show_volume_control', $form_state->getValue('show_volume_control'))
      ->set('show_download_button', $form_state->getValue('show_download_button'))
      ->set('custom_template', $form_state->getValue('custom_template'))
      ->set('storage_type', $storage_type)
      ->set('storage_settings', $storage_settings)
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Get available content types.
   *
   * @return array
   *   Array of content type options.
   */
  protected function getContentTypes(): array {
    $content_types = [];

    try {
      $types = $this->entityTypeManager
        ->getStorage('node_type')
        ->loadMultiple();

      foreach ($types as $type) {
        $content_types[$type->id()] = $type->label();
      }
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Unable to load content types.'));
    }

    return $content_types;
  }

}
