<?php

namespace Drupal\tts\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\tts\Service\TextToSpeechService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\node\NodeInterface;

/**
 * Provides a TTS Audio Player block.
 *
 * @Block(
 *   id = "tts_player",
 *   admin_label = @Translation("TTS Audio Player"),
 *   category = @Translation("AI")
 * )
 */
class TtsPlayerBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The TTS service.
   *
   * @var \Drupal\tts\Service\TextToSpeechService
   */
  protected $ttsService;

  /**
   * Constructs a TtsPlayerBlock object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\tts\Service\TextToSpeechService $tts_service
   *   The TTS service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    RouteMatchInterface $route_match,
    ConfigFactoryInterface $config_factory,
    TextToSpeechService $tts_service,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->routeMatch = $route_match;
    $this->configFactory = $config_factory;
    $this->ttsService = $tts_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_route_match'),
      $container->get('config.factory'),
      $container->get('tts.text_to_speech')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $config = $this->configFactory->get('tts.settings');

    // Check if player should be shown.
    if (!$config->get('show_player_block')) {
      return [];
    }

    // Get current node.
    $node = $this->routeMatch->getParameter('node');

    if (!$node instanceof NodeInterface) {
      return [];
    }

    // Check if content type is enabled.
    $enabled_types = $config->get('enabled_content_types') ?? [];
    if (!in_array($node->bundle(), $enabled_types)) {
      return [];
    }

    // Check access.
    if (!$node->access('view')) {
      return [];
    }

    // Get audio URL.
    $audio_url = $this->ttsService->getAudioUrl($node);

    if (!$audio_url) {
      // Show loading state or generate button.
      return [
        '#theme' => 'tts_player',
        '#node_id' => $node->id(),
        '#audio_url' => NULL,
        '#loading' => TRUE,
        '#attached' => [
          'library' => [
            'tts/player',
          ],
        ],
        '#cache' => [
          'contexts' => ['url.path'],
          'tags' => ['node:' . $node->id()],
        ],
      ];
    }

    return [
      '#theme' => 'tts_player',
      '#node_id' => $node->id(),
      '#audio_url' => $audio_url,
      '#title' => $node->getTitle(),
      '#loading' => FALSE,
      '#attached' => [
        'library' => [
          'tts/player',
        ],
      ],
      '#cache' => [
        'contexts' => ['url.path'],
        'tags' => ['node:' . $node->id()],
        'max-age' => $config->get('cache_duration') ?? 604800,
      ],
    ];
  }

}
