<?php

namespace Drupal\tts\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\FileSystemInterface;
use Psr\Log\LoggerInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;

/**
 * Service for managing external storage uploads.
 */
class StorageManager {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * Constructs a StorageManager object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    FileSystemInterface $file_system,
    LoggerInterface $logger,
    ClientInterface $http_client,
  ) {
    $this->configFactory = $config_factory;
    $this->fileSystem = $file_system;
    $this->logger = $logger;
    $this->httpClient = $http_client;
  }

  /**
   * Upload file to configured storage backend.
   *
   * @param string $file_data
   *   The file binary data.
   * @param string $filename
   *   The filename.
   * @param string $content_type
   *   The content type (e.g., 'audio/mpeg').
   *
   * @return string|null
   *   The public URL of the uploaded file, or NULL on failure.
   */
  public function uploadFile(string $file_data, string $filename, string $content_type = 'audio/mpeg'): ?string {
    $config = $this->configFactory->get('tts.settings');
    $storage_type = $config->get('storage_type') ?: 'local';

    try {
      switch ($storage_type) {
        case 's3':
          return $this->uploadToS3($file_data, $filename, $content_type);

        case 'azure':
          return $this->uploadToAzure($file_data, $filename, $content_type);

        case 'gcs':
          return $this->uploadToGcs($file_data, $filename, $content_type);

        case 'cdn':
          return $this->uploadToCdn($file_data, $filename, $content_type);

        case 'local':
        default:
          // Local storage is handled elsewhere.
          return NULL;
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Storage upload failed: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Upload file to AWS S3 or S3-compatible storage.
   *
   * @param string $file_data
   *   The file binary data.
   * @param string $filename
   *   The filename.
   * @param string $content_type
   *   The content type.
   *
   * @return string|null
   *   The public URL or NULL on failure.
   */
  protected function uploadToS3(string $file_data, string $filename, string $content_type): ?string {
    $config = $this->configFactory->get('tts.settings');
    $settings = $config->get('storage_settings.s3');

    $bucket = $settings['bucket'] ?? '';
    $region = $settings['region'] ?? 'us-east-1';
    $access_key = $settings['access_key'] ?? '';
    $secret_key = $settings['secret_key'] ?? '';
    $endpoint = $settings['endpoint'] ?? '';
    $custom_domain = $settings['custom_domain'] ?? '';

    if (empty($bucket) || empty($access_key) || empty($secret_key)) {
      $this->logger->error('S3 credentials not configured.');
      return NULL;
    }

    // Build S3 URL.
    if (!empty($endpoint)) {
      $host = parse_url($endpoint, PHP_URL_HOST);
      $url = $endpoint . '/' . $bucket . '/' . $filename;
    }
    else {
      $host = $bucket . '.s3.' . $region . '.amazonaws.com';
      $url = 'https://' . $host . '/' . $filename;
    }

    // Create AWS Signature Version 4.
    $timestamp = gmdate('Ymd\THis\Z');
    $datestamp = gmdate('Ymd');
    $service = 's3';

    $canonical_uri = '/' . $filename;
    $canonical_querystring = '';
    $canonical_headers = 'host:' . $host . "\n" . 'x-amz-content-sha256:UNSIGNED-PAYLOAD' . "\n" . 'x-amz-date:' . $timestamp . "\n";
    $signed_headers = 'host;x-amz-content-sha256;x-amz-date';
    $payload_hash = 'UNSIGNED-PAYLOAD';

    $canonical_request = "PUT\n" . $canonical_uri . "\n" . $canonical_querystring . "\n" . $canonical_headers . "\n" . $signed_headers . "\n" . $payload_hash;
    $algorithm = 'AWS4-HMAC-SHA256';
    $credential_scope = $datestamp . '/' . $region . '/' . $service . '/aws4_request';
    $string_to_sign = $algorithm . "\n" . $timestamp . "\n" . $credential_scope . "\n" . hash('sha256', $canonical_request);

    $signing_key = $this->getSignatureKey($secret_key, $datestamp, $region, $service);
    $signature = hash_hmac('sha256', $string_to_sign, $signing_key);

    $authorization_header = $algorithm . ' Credential=' . $access_key . '/' . $credential_scope . ', SignedHeaders=' . $signed_headers . ', Signature=' . $signature;

    try {
      $this->httpClient->request('PUT', $url, [
        'body' => $file_data,
        'headers' => [
          'Host' => $host,
          'Content-Type' => $content_type,
          'x-amz-content-sha256' => $payload_hash,
          'x-amz-date' => $timestamp,
          'x-amz-acl' => 'public-read',
          'Authorization' => $authorization_header,
        ],
      ]);

      // Return custom domain URL if configured, otherwise S3 URL.
      if (!empty($custom_domain)) {
        return rtrim($custom_domain, '/') . '/' . $filename;
      }
      return $url;
    }
    catch (RequestException $e) {
      $this->logger->error('S3 upload failed: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * Get AWS Signature Key.
   */
  protected function getSignatureKey($key, $dateStamp, $regionName, $serviceName) {
    $kDate = hash_hmac('sha256', $dateStamp, 'AWS4' . $key, TRUE);
    $kRegion = hash_hmac('sha256', $regionName, $kDate, TRUE);
    $kService = hash_hmac('sha256', $serviceName, $kRegion, TRUE);
    $kSigning = hash_hmac('sha256', 'aws4_request', $kService, TRUE);
    return $kSigning;
  }

  /**
   * Upload file to Azure Blob Storage.
   *
   * @param string $file_data
   *   The file binary data.
   * @param string $filename
   *   The filename.
   * @param string $content_type
   *   The content type.
   *
   * @return string|null
   *   The public URL or NULL on failure.
   */
  protected function uploadToAzure(string $file_data, string $filename, string $content_type): ?string {
    $config = $this->configFactory->get('tts.settings');
    $settings = $config->get('storage_settings.azure');

    $account_name = $settings['account_name'] ?? '';
    $account_key = $settings['account_key'] ?? '';
    $container = $settings['container'] ?? '';
    $custom_domain = $settings['custom_domain'] ?? '';

    if (empty($account_name) || empty($account_key) || empty($container)) {
      $this->logger->error('Azure credentials not configured.');
      return NULL;
    }

    $url = 'https://' . $account_name . '.blob.core.windows.net/' . $container . '/' . $filename;
    $date = gmdate('D, d M Y H:i:s \G\M\T');
    $content_length = strlen($file_data);

    $string_to_sign = "PUT\n\n$content_type\n\n\n\n\n\n\n\n\n\nx-ms-blob-type:BlockBlob\nx-ms-date:$date\nx-ms-version:2019-12-12\n/$account_name/$container/$filename";
    $signature = base64_encode(hash_hmac('sha256', $string_to_sign, base64_decode($account_key), TRUE));

    try {
      $this->httpClient->request('PUT', $url, [
        'body' => $file_data,
        'headers' => [
          'Content-Type' => $content_type,
          'Content-Length' => $content_length,
          'x-ms-blob-type' => 'BlockBlob',
          'x-ms-date' => $date,
          'x-ms-version' => '2019-12-12',
          'Authorization' => 'SharedKey ' . $account_name . ':' . $signature,
        ],
      ]);

      if (!empty($custom_domain)) {
        return rtrim($custom_domain, '/') . '/' . $filename;
      }
      return $url;
    }
    catch (RequestException $e) {
      $this->logger->error('Azure upload failed: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * Upload file to Google Cloud Storage.
   *
   * @param string $file_data
   *   The file binary data.
   * @param string $filename
   *   The filename.
   * @param string $content_type
   *   The content type.
   *
   * @return string|null
   *   The public URL or NULL on failure.
   */
  protected function uploadToGcs(string $file_data, string $filename, string $content_type): ?string {
    $config = $this->configFactory->get('tts.settings');
    $settings = $config->get('storage_settings.gcs');

    $bucket = $settings['bucket'] ?? '';
    $credentials_json = $settings['credentials_json'] ?? '';
    $custom_domain = $settings['custom_domain'] ?? '';

    if (empty($bucket) || empty($credentials_json)) {
      $this->logger->error('GCS credentials not configured.');
      return NULL;
    }

    // For GCS, you would typically use the Google Cloud SDK.
    // This is a simplified implementation using the JSON API.
    $credentials = json_decode($credentials_json, TRUE);
    if (!$credentials || !isset($credentials['private_key'])) {
      $this->logger->error('Invalid GCS credentials JSON.');
      return NULL;
    }

    $url = 'https://storage.googleapis.com/upload/storage/v1/b/' . $bucket . '/o?uploadType=media&name=' . urlencode($filename);

    // Create JWT token for authentication.
    $jwt = $this->createGcsJwt($credentials);
    if (!$jwt) {
      return NULL;
    }

    try {
      $this->httpClient->request('POST', $url, [
        'body' => $file_data,
        'headers' => [
          'Content-Type' => $content_type,
          'Authorization' => 'Bearer ' . $jwt,
        ],
      ]);

      if (!empty($custom_domain)) {
        return rtrim($custom_domain, '/') . '/' . $filename;
      }
      return 'https://storage.googleapis.com/' . $bucket . '/' . $filename;
    }
    catch (RequestException $e) {
      $this->logger->error('GCS upload failed: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * Create JWT token for GCS authentication.
   *
   * @param array $credentials
   *   The GCS credentials array.
   *
   * @return string|null
   *   The JWT token or NULL on failure.
   */
  protected function createGcsJwt(array $credentials): ?string {
    // Simplified JWT creation - in production, use a JWT library.
    $now = time();
    $header = json_encode(['alg' => 'RS256', 'typ' => 'JWT']);
    $payload = json_encode([
      'iss' => $credentials['client_email'],
      'scope' => 'https://www.googleapis.com/auth/devstorage.read_write',
      'aud' => 'https://oauth2.googleapis.com/token',
      'exp' => $now + 3600,
      'iat' => $now,
    ]);

    $base64_header = rtrim(strtr(base64_encode($header), '+/', '-_'), '=');
    $base64_payload = rtrim(strtr(base64_encode($payload), '+/', '-_'), '=');
    $signature_input = $base64_header . '.' . $base64_payload;

    openssl_sign($signature_input, $signature, $credentials['private_key'], OPENSSL_ALGO_SHA256);
    $base64_signature = rtrim(strtr(base64_encode($signature), '+/', '-_'), '=');

    return $signature_input . '.' . $base64_signature;
  }

  /**
   * Upload file to generic CDN with API.
   *
   * @param string $file_data
   *   The file binary data.
   * @param string $filename
   *   The filename.
   * @param string $content_type
   *   The content type.
   *
   * @return string|null
   *   The public URL or NULL on failure.
   */
  protected function uploadToCdn(string $file_data, string $filename, string $content_type): ?string {
    $config = $this->configFactory->get('tts.settings');
    $settings = $config->get('storage_settings.cdn');

    $base_url = $settings['base_url'] ?? '';
    $api_endpoint = $settings['api_endpoint'] ?? '';
    $api_key = $settings['api_key'] ?? '';

    if (empty($base_url) || empty($api_endpoint) || empty($api_key)) {
      $this->logger->error('CDN credentials not configured.');
      return NULL;
    }

    try {
      $response = $this->httpClient->request('POST', $api_endpoint, [
        'multipart' => [
          [
            'name' => 'file',
            'contents' => $file_data,
            'filename' => $filename,
          ],
        ],
        'headers' => [
          'Authorization' => 'Bearer ' . $api_key,
        ],
      ]);

      $result = json_decode($response->getBody()->getContents(), TRUE);

      // Assuming the API returns the filename or path.
      if (isset($result['path'])) {
        return rtrim($base_url, '/') . '/' . ltrim($result['path'], '/');
      }

      return rtrim($base_url, '/') . '/' . $filename;
    }
    catch (RequestException $e) {
      $this->logger->error('CDN upload failed: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }

}
