<?php

namespace Drupal\tts\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\node\NodeInterface;
use Drupal\Component\Plugin\PluginManagerInterface;
use Drupal\ai\OperationType\TextToSpeech\TextToSpeechInput;

/**
 * Service for Text-to-Speech operations using AI providers.
 */
class TextToSpeechService {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * The AI provider manager.
   *
   * @var \Drupal\Component\Plugin\PluginManagerInterface|null
   */
  protected $aiProviderManager;

  /**
   * The storage manager.
   *
   * @var \Drupal\tts\Service\StorageManager
   */
  protected $storageManager;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The file URL generator.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected $fileUrlGenerator;

  /**
   * Constructs a TextToSpeechService object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   * @param \Drupal\Component\Plugin\PluginManagerInterface $ai_provider_manager
   *   The AI provider manager.
   * @param \Drupal\tts\Service\StorageManager $storage_manager
   *   The storage manager.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\Core\File\FileUrlGeneratorInterface $file_url_generator
   *   The file URL generator.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    EntityTypeManagerInterface $entity_type_manager,
    FileSystemInterface $file_system,
    LoggerChannelFactoryInterface $logger_factory,
    CacheBackendInterface $cache,
    PluginManagerInterface $ai_provider_manager,
    StorageManager $storage_manager,
    ModuleHandlerInterface $module_handler,
    FileUrlGeneratorInterface $file_url_generator,
  ) {
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->fileSystem = $file_system;
    $this->logger = $logger_factory->get('tts');
    $this->cache = $cache;
    $this->aiProviderManager = $ai_provider_manager;
    $this->storageManager = $storage_manager;
    $this->moduleHandler = $module_handler;
    $this->fileUrlGenerator = $file_url_generator;
  }

  /**
   * Generate speech audio from text.
   *
   * @param string $text
   *   The text to convert to speech.
   * @param array $options
   *   Optional configuration overrides.
   *
   * @return array|null
   *   Array containing audio data and metadata, or NULL on failure.
   */
  public function generateSpeech(string $text, array $options = []): ?array {
    $config = $this->configFactory->get('tts.settings');

    // Clean and prepare text.
    $text = $this->prepareText($text);

    if (empty($text)) {
      $this->logger->warning('Empty text provided for TTS generation.');
      return NULL;
    }

    // Check text length limit.
    $max_length = $config->get('max_text_length') ?? 4096;
    if (strlen($text) > $max_length) {
      $text = substr($text, 0, $max_length);
      $this->logger->notice('Text truncated to @length characters for TTS generation.', ['@length' => $max_length]);
    }

    // Generate cache key.
    $cache_key = $this->generateCacheKey($text, $options);

    // Check cache first.
    if ($cached = $this->cache->get($cache_key)) {
      return $cached->data;
    }

    // Merge options with configuration.
    $tts_config = [
      'voice' => $options['voice'] ?? $config->get('voice_model'),
      'format' => $options['format'] ?? $config->get('audio_format'),
    ];

    try {
      // Generate audio using AI provider.
      $audio_data = $this->generateAudioWithProvider($text, $tts_config);

      if (!$audio_data) {
        return NULL;
      }

      // Save audio to file (with node_id if provided in options).
      $node_id = $options['node_id'] ?? NULL;
      $file_uri = $this->saveAudioFile($audio_data, $tts_config['format'], $node_id);

      if (!$file_uri) {
        return NULL;
      }

      $result = [
        'uri' => $file_uri,
        'url' => $this->getPublicUrl($file_uri),
        'format' => $tts_config['format'],
        'text' => $text,
        'duration' => $this->estimateDuration($text),
        'generated' => time(),
      ];

      // Cache the result.
      $cache_duration = $config->get('cache_duration') ?? 604800;
      $this->cache->set($cache_key, $result, time() + $cache_duration);

      return $result;
    }
    catch (\Exception $e) {
      $this->logger->error('Error generating TTS audio: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * Generate speech from a node's content.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node to convert to speech.
   * @param array $options
   *   Optional configuration overrides.
   *
   * @return array|null
   *   Array containing audio data and metadata, or NULL on failure.
   */
  public function generateNodeSpeech(NodeInterface $node, array $options = []): ?array {
    $config = $this->configFactory->get('tts.settings');

    // Check if content type is enabled.
    $enabled_types = $config->get('enabled_content_types') ?? [];
    if (!in_array($node->bundle(), $enabled_types)) {
      $this->logger->notice('Content type @type is not enabled for TTS.', ['@type' => $node->bundle()]);
      return NULL;
    }

    // Check if audio file already exists for this node.
    $existing_audio = $this->getExistingNodeAudio($node);
    if ($existing_audio) {
      // File exists and is valid, return it without regenerating.
      return $existing_audio;
    }

    // Extract text from node.
    $text = $this->extractNodeText($node);

    if (empty($text)) {
      $this->logger->warning('No text content found in node @nid.', ['@nid' => $node->id()]);
      return NULL;
    }

    // Add node context to options.
    $options['node_id'] = $node->id();
    $options['node_type'] = $node->bundle();

    return $this->generateSpeech($text, $options);
  }

  /**
   * Get audio URL for a node.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node.
   *
   * @return string|null
   *   The audio URL or NULL if not available.
   */
  public function getAudioUrl(NodeInterface $node): ?string {
    $audio_data = $this->generateNodeSpeech($node);
    return $audio_data['url'] ?? NULL;
  }

  /**
   * Extract text content from a node.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node.
   *
   * @return string
   *   The extracted text.
   */
  protected function extractNodeText(NodeInterface $node): string {
    $config = $this->configFactory->get('tts.settings');
    $included_fields = $config->get('included_fields') ?? ['body'];

    $text_parts = [];

    // Add title.
    $text_parts[] = $node->getTitle();

    // Extract field content.
    foreach ($included_fields as $field_name) {
      if ($node->hasField($field_name) && !$node->get($field_name)->isEmpty()) {
        $field_value = $node->get($field_name)->value;
        if (!empty($field_value)) {
          $text_parts[] = $field_value;
        }
      }
    }

    return implode("\n\n", $text_parts);
  }

  /**
   * Prepare text for TTS processing.
   *
   * @param string $text
   *   The raw text.
   *
   * @return string
   *   The cleaned text.
   */
  protected function prepareText(string $text): string {
    // Strip HTML tags.
    $text = strip_tags($text);

    // Decode HTML entities.
    $text = html_entity_decode($text, ENT_QUOTES | ENT_HTML5, 'UTF-8');

    // Remove extra whitespace.
    $text = preg_replace('/\s+/', ' ', $text);

    // Trim.
    $text = trim($text);

    // Allow modules to alter the text.
    $this->moduleHandler->alter('tts_text', $text);

    return $text;
  }

  /**
   * Generate audio using the configured AI provider.
   *
   * @param string $text
   *   The text to convert.
   * @param array $config
   *   The TTS configuration.
   *
   * @return string|null
   *   The audio data or NULL on failure.
   */
  protected function generateAudioWithProvider(string $text, array $config): ?string {
    try {
      // Get the AI module's default TTS provider and model.
      $ai_config = $this->configFactory->get('ai.settings');
      $tts_provider = $ai_config->get('default_providers.text_to_speech.provider_id');
      $tts_model = $ai_config->get('default_providers.text_to_speech.model_id');

      if (empty($tts_provider) || empty($tts_model)) {
        $this->logger->error('No TTS provider/model configured in AI module. Please configure at /admin/config/ai/settings');
        return NULL;
      }

      // Create TTS input.
      $input = new TextToSpeechInput($text);

      // Build provider configuration with voice setting.
      $provider_config = [];
      if (!empty($config['voice'])) {
        $provider_config['voice'] = $config['voice'];
      }

      // Get provider instance with configuration.
      $provider = $this->aiProviderManager->createInstance($tts_provider, $provider_config);

      // Generate audio using AI module's TTS interface.
      // The provider is wrapped in a ProviderProxy that uses __call()
      // magic method.
      $output = $provider->textToSpeech($input, $tts_model);

      // Get the audio data from output.
      $audio_files = $output->getNormalized();
      if (empty($audio_files)) {
        $this->logger->error('No audio generated from TTS provider.');
        return NULL;
      }

      // Get the first audio file.
      $audio_file = reset($audio_files);
      return $audio_file->getBinary();

    }
    catch (\Exception $e) {
      $this->logger->error('TTS generation error: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * Save audio data to file.
   *
   * @param string $audio_data
   *   The audio data.
   * @param string $format
   *   The audio format.
   * @param int|null $node_id
   *   Optional node ID to associate with the file.
   *
   * @return string|null
   *   The file URI or NULL on failure.
   */
  protected function saveAudioFile(string $audio_data, string $format, ?int $node_id = NULL): ?string {
    try {
      $config = $this->configFactory->get('tts.settings');
      $storage_type = $config->get('storage_type') ?: 'local';

      // Generate unique filename with node ID if provided.
      if ($node_id) {
        $filename = 'tts_node_' . $node_id . '_' . uniqid() . '.' . $format;
      }
      else {
        $filename = uniqid('tts_', TRUE) . '.' . $format;
      }

      // Check if using external storage.
      if ($storage_type !== 'local') {
        $content_type = $this->getContentType($format);
        $external_url = $this->storageManager->uploadFile($audio_data, $filename, $content_type);

        if ($external_url) {
          $this->logger->info('TTS audio uploaded to external storage: @url', ['@url' => $external_url]);
          return $external_url;
        }

        // Fall back to local storage if external upload fails.
        $this->logger->warning('External storage upload failed, falling back to local storage.');
      }

      // Local storage (default or fallback).
      $directory = 'public://tts';
      $this->fileSystem->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);

      $uri = $directory . '/' . $filename;

      // Save file locally.
      $result = $this->fileSystem->saveData($audio_data, $uri, FileSystemInterface::EXISTS_REPLACE);

      if ($result) {
        $this->logger->info('TTS audio saved to @uri', ['@uri' => $uri]);
        return $result;
      }

      return NULL;
    }
    catch (\Exception $e) {
      $this->logger->error('Error saving audio file: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * Get content type from audio format.
   *
   * @param string $format
   *   The audio format (mp3, ogg, wav, etc.).
   *
   * @return string
   *   The MIME type.
   */
  protected function getContentType(string $format): string {
    $content_types = [
      'mp3' => 'audio/mpeg',
      'ogg' => 'audio/ogg',
      'wav' => 'audio/wav',
      'flac' => 'audio/flac',
      'aac' => 'audio/aac',
      'm4a' => 'audio/mp4',
    ];

    return $content_types[strtolower($format)] ?? 'audio/mpeg';
  }

  /**
   * Get public URL for a file URI.
   *
   * @param string $uri
   *   The file URI.
   *
   * @return string
   *   The public URL.
   */
  protected function getPublicUrl(string $uri): string {
    return $this->fileUrlGenerator->generateAbsoluteString($uri);
  }

  /**
   * Generate cache key for text and options.
   *
   * @param string $text
   *   The text.
   * @param array $options
   *   The options.
   *
   * @return string
   *   The cache key.
   */
  protected function generateCacheKey(string $text, array $options): string {
    $key_data = [
      'text' => $text,
      'options' => $options,
    ];
    return 'tts:' . md5(serialize($key_data));
  }

  /**
   * Estimate audio duration based on text length.
   *
   * @param string $text
   *   The text.
   *
   * @return float
   *   Estimated duration in seconds.
   */
  protected function estimateDuration(string $text): float {
    // Average speaking rate is about 150 words per minute.
    $word_count = str_word_count($text);
    return ($word_count / 150) * 60;
  }

  /**
   * Clear TTS cache.
   *
   * @param string|null $cache_key
   *   Optional specific cache key to clear.
   */
  public function clearCache(?string $cache_key = NULL): void {
    if ($cache_key) {
      $this->cache->delete($cache_key);
    }
    else {
      $this->cache->deleteAll();
    }
  }

  /**
   * Get existing audio file for a node if valid.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node.
   *
   * @return array|null
   *   Audio data array if valid file exists, NULL otherwise.
   */
  protected function getExistingNodeAudio(NodeInterface $node): ?array {
    try {
      $config = $this->configFactory->get('tts.settings');
      $format = $config->get('audio_format') ?? 'mp3';
      $directory = 'public://tts';

      // Check if directory exists.
      if (!is_dir($this->fileSystem->realpath($directory))) {
        return NULL;
      }

      // Find existing audio files for this node.
      $pattern = $this->fileSystem->realpath($directory) . '/tts_node_' . $node->id() . '_*.' . $format;
      $files = glob($pattern);

      if (empty($files)) {
        return NULL;
      }

      // Get the most recent file.
      usort($files, function ($a, $b) {
        return filemtime($b) - filemtime($a);
      });
      $latest_file = reset($files);

      // Check if node has been modified since file was created.
      $file_time = filemtime($latest_file);
      $node_changed = $node->getChangedTime();

      if ($node_changed > $file_time) {
        // Node has been modified, file is outdated.
        // Delete the old file(s).
        foreach ($files as $old_file) {
          @unlink($old_file);
        }
        return NULL;
      }

      // File is valid, return its info.
      $filename = basename($latest_file);
      $uri = $directory . '/' . $filename;

      // Extract text for duration estimation.
      $text = $this->extractNodeText($node);

      return [
        'uri' => $uri,
        'url' => $this->getPublicUrl($uri),
        'format' => $format,
        'text' => $text,
        'duration' => $this->estimateDuration($text),
        'generated' => $file_time,
        'reused' => TRUE,
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Error checking existing audio: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * Clean up old audio files.
   *
   * @param int $age
   *   Maximum age in seconds.
   */
  public function cleanupOldFiles(int $age = 604800): void {
    try {
      $directory = 'public://tts';
      $files = $this->fileSystem->scanDirectory($directory, '/^tts_.*\.(mp3|ogg|wav)$/');

      $deleted = 0;
      $current_time = time();

      foreach ($files as $file) {
        $file_age = $current_time - filemtime($file->uri);
        if ($file_age > $age) {
          if ($this->fileSystem->delete($file->uri)) {
            $deleted++;
          }
        }
      }

      if ($deleted > 0) {
        $this->logger->info('Cleaned up @count old TTS audio files.', ['@count' => $deleted]);
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Error cleaning up old files: @message', ['@message' => $e->getMessage()]);
    }
  }

}
