<?php

namespace Drupal\tts;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Security\TrustedCallbackInterface;
use Drupal\tts\Service\TextToSpeechService;

/**
 * Lazy builder for TTS player.
 */
class TtsLazyBuilder implements TrustedCallbackInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The TTS service.
   *
   * @var \Drupal\tts\Service\TextToSpeechService
   */
  protected $ttsService;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a TtsLazyBuilder object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\tts\Service\TextToSpeechService $tts_service
   *   The TTS service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    TextToSpeechService $tts_service,
    ConfigFactoryInterface $config_factory,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->ttsService = $tts_service;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks() {
    return ['renderPlayer'];
  }

  /**
   * Lazy builder callback for rendering the TTS player.
   *
   * @param int $node_id
   *   The node ID.
   *
   * @return array
   *   Render array for the TTS player.
   */
  public function renderPlayer($node_id) {
    // Load the node.
    $node = $this->entityTypeManager->getStorage('node')->load($node_id);

    if (!$node) {
      return [];
    }

    // Check access.
    if (!$node->access('view')) {
      return [];
    }

    // Get audio URL.
    $audio_url = $this->ttsService->getAudioUrl($node);

    /** @var \Drupal\node\NodeInterface $node */
    $title = $node->getTitle();

    // Get configuration.
    $config = $this->configFactory->get('tts.settings');

    // Check for custom template override.
    $custom_template = $config->get('custom_template');
    $theme = !empty($custom_template) ? $custom_template : 'tts_player';

    // Get boolean values properly (handle NULL with defaults).
    $show_download_button = $config->get('show_download_button') !== NULL ? (bool) $config->get('show_download_button') : TRUE;
    $show_volume_control = $config->get('show_volume_control') !== NULL ? (bool) $config->get('show_volume_control') : TRUE;

    // Return render array.
    return [
      '#theme' => $theme,
      '#node_id' => $node->id(),
      '#audio_url' => $audio_url,
      '#title' => $title,
      '#loading' => empty($audio_url),
      // Player customization settings.
      '#player_style' => $config->get('player_style') ?: 'default',
      '#show_download_button' => $show_download_button,
      '#show_volume_control' => $show_volume_control,
      '#attached' => [
        'library' => [
          'tts/player',
        ],
      ],
      '#cache' => [
        'contexts' => ['url.path'],
        'tags' => ['node:' . $node->id(), 'config:tts.settings'],
      ],
    ];
  }

}
