<?php

namespace Drupal\tugboat;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\State\StateInterface;
use Psr\Log\LoggerInterface;

class Cron {

  protected $client;
  protected $configFactory;
  protected $logger;
  protected $state;
  protected $time;

  public function __construct(TugboatClient $client, ConfigFactoryInterface $config_factory, LoggerInterface $logger, StateInterface $state, TimeInterface $time) {
    $this->client = $client;
    $this->configFactory = $config_factory;
    $this->logger = $logger;
    $this->state = $state;
    $this->time = $time;
  }

  public function run() {
    $config = $this->configFactory->get('tugboat.settings');

    // Run on an interval that matches the length of the current lifetime.
    $request_time = $this->time->getRequestTime();
    $lifetime_timestamp = $config->get('sandbox_lifetime');
    if ($request_time - $this->state->get('tugboat.cron', 0) <= $lifetime_timestamp) {
      return;
    }

    $delete_older_than_timestamp = $request_time - $lifetime_timestamp;
    $repository_id = $config->get('repository_id');
    try {
      $existing_previews = $this->client->requestWithApiKey('GET', "repos/$repository_id/previews");
    } catch (\Exception $e) {
      $this->logger->error('Unable to select sandboxes during cron run. Error from tugboat: %error', [
        '%error' => $e->getMessage(),
      ]);
      throw $e;
    }

    $this->state->set('tugboat.cron', $request_time);

    $return_data = Json::decode((string) $existing_previews->getBody());

    $delete_count = 0;
    foreach ($return_data as $sandbox) {
      // Delete if not a base (anchor) preview and it's past the expire date.
      if (!$sandbox['anchor'] && strtotime($sandbox['createdAt']) < $delete_older_than_timestamp) {
        $sandbox_id = $sandbox['id'];
        try {
          $this->client->requestWithApiKey('DELETE', "previews/$sandbox_id");
          $delete_count++;
        } catch (\Exception $e) {
          $this->logger->error('Unable to delete individual sandbox during cron run. Error from tugboat: %error', [
            '%error' => $e->getMessage(),
          ]);
        }
      }
    }

    if ($delete_count) {
      $this->logger->notice(
        'Deleted %count demo sandboxes older than %date.',
        [
          '%count' => $delete_count,
          '%date' => date('D, m/d/Y - H:i', $delete_older_than_timestamp)
        ]);
    }
  }

}
