<?php

namespace Drupal\tugboat\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\tugboat\TugboatClient;
use Drupal\tugboat\TugboatExecute;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Tugboat QA API settings for this site.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * The Tugboat Execute service.
   *
   * @var \Drupal\tugboat\TugboatExecute
   */
  protected $tugboatExecute;

  /**
   * The Tugboat client.
   *
   * @var \Drupal\tugboat\TugboatClient
   */
  protected $tugboatClient;

  public function __construct(ConfigFactoryInterface $config_factory, TugboatExecute $execute, TugboatClient $client) {
    parent::__construct($config_factory);
    $this->tugboatExecute = $execute;
    $this->tugboatClient = $client;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('tugboat.execute'),
      $container->get('tugboat.client')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'tugboat_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['tugboat.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $settings = $this->config('tugboat.settings');
    $settings_with_overrides = $this->configFactory()->get('tugboat.settings');

    $tugboat_token = $settings_with_overrides->get('token');
    $masked_token = $tugboat_token
      ? substr($tugboat_token, 0, 4) . str_repeat("*", strlen($tugboat_token) - 8) . substr($tugboat_token, -4)
      : '';

    $form['token'] = [
      '#type' => 'item',
      '#title' => $this->t('Tugboat Secret Token'),
      '#markup' => $tugboat_token ? $masked_token : $this->t('Not found! Must be set in settings.php!'),
      '#description' => $this->t("Provides API access to tugboat.qa. This setting must be stored in settings.php as <code>\$config['tugboat.settings']['token']</code>"),
    ];

    $form['executable_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Tugboat Executable Path'),
      '#default_value' => $settings->get('executable_path'),
      '#description' => $this->t('The Tugboat executable binary file location on the server or relative to the Drupal installation. This file is downloadable from <a href=":url">https://dashboard2.tugboat.qa/downloads</a>', [':url' => 'https://dashboard2.tugboat.qa/downloads']),
    ];

    $form['repository_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Tugboat Repository ID'),
      '#default_value' => $settings->get('repository_id'),
      '#description' => $this->t('The repository ID as provided by the Tugboat dashboard. A 24-character alphanumeric hash, such as <code>5bdb5c268eabd5000137a87b</code>.'),
    ];

    $form['repository_base'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Tugboat Repository Base Name'),
      '#default_value' => $settings->get('repository_base'),
      '#description' => $this->t('The branch, tag, or pull request name from which to base the previews. A preview with a matching name must be specified in the Tugboat dashboard. Usually <code>master</code> is the latest version.'),
    ];

    $form['sandbox_lifetime'] = [
      '#type' => 'select',
      '#title' => $this->t('Sandbox lifetime'),
      '#default_value' => $settings->get('sandbox_lifetime'),
      '#description' => $this->t('The amount of time the Tugboat preview will be available. Previews older than this will automatically be torn down on cron jobs.'),
      '#options' => [
        '7200' => '2 hours',
        '14400' => '4 hours',
        '28800' =>   '8 hours',
        '86400' => '1 day',
        '172800' =>  '2 days',
        '259200' =>  '3 days',
        '345600' => '4 days',
        '432000' =>  '5 days',
        '518400' =>  '6 days',
        '604800' => '1 week',
        '1209600' => '2 weeks',
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $error_string = '';
    // Belt and suspenders: do not save the token in config.
    $form_state->setValue('token', '');

    // Validate the path to the tugboat executable.
    $executable_path = $form_state->getValue('executable_path');
    if ($executable_path && !$this->tugboatExecute->isExecutable($executable_path, $error_string)) {
      $form_state->setErrorByName('executable_path', $error_string);
    }

    // Validate the repository. This will fail too if the executable does not work.
    $repo = $form_state->getValue('repository_id');
    //$branch = $form_state->getValue('repository_base');
    try {
      $this->tugboatClient->requestWithApiKey('GET', "repos/$repo");
    } catch (\Throwable $e) {
      $message = $this->t(
          'The provided repository ID was not found. Tugboat returned the response, "%response".',
          ['%response' => $e->getMessage()]
        );
      $form_state->setErrorByName('repository_id', $message);
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('tugboat.settings')
      ->set('token', '')
      ->set('executable_path', $form_state->getValue('executable_path'))
      ->set('repository_id', $form_state->getValue('repository_id'))
      ->set('repository_base', $form_state->getValue('repository_base'))
      ->set('sandbox_lifetime', $form_state->getValue('sandbox_lifetime'))
      ->save();
    parent::submitForm($form, $form_state);
  }

}
