<?php

namespace Drupal\tugboat;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Http\ClientFactory;

/**
 * API Client for the Tugboat API.
 */
class TugboatClient {

  const API_URL = 'https://api.tugboatqa.com/v3/';

  protected $client;
  protected $settings;

  public function __construct(ClientFactory $client_factory, ConfigFactoryInterface $config_factory) {
    $this->client = $client_factory->fromOptions([
      'base_uri' => self::API_URL,
      'headers' => [
        'Accept' => 'application/json',
      ],
    ]);
    $this->settings = $config_factory->get('tugboat.settings');
  }

  /**
   * Perform a request using the API Key.
   *
   * Requests with API Keys are read-write.
   *
   * @param string $method
   *   The method.
   * @param string $uri
   *   The URI.
   *
   * @return \Psr\Http\Message\ResponseInterface
   *   The response.
   *
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  public function requestWithApiKey($method, $uri, array $payload = []) {
    $request_options = [
      'headers' => [
        'Authorization' => sprintf('Bearer %s', $this->settings->get('token')),
      ]
    ];
    if ($payload !== []) {
      if ($method === 'GET') {
        throw new \InvalidArgumentException('Cannot set a body with a GET request');
      }
      $request_options['json'] = $payload;
    }

    return $this->client->request($method, $uri, $request_options);
  }

  /**
   * Perform a request using a URL token.
   *
   * Requests with URL tokens are read only.
   *
   * @param string $method
   *   The method.
   * @param string $uri
   *   The URI.
   * @param string $token
   *   The token.
   *
   * @return \Psr\Http\Message\ResponseInterface
   *   The response.
   *
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  public function requestWithUrlToken($method, $uri, $token) {
    return $this->client->request($method, $uri, [
      'headers' => [
        'Authorization' => sprintf('Authorization: Bearer %s', $token),
      ]
    ]);
  }

}
