<?php

namespace Drupal\tugboat;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * TugboatExecute service.
 */
class TugboatExecute {
    use StringTranslationTrait;

  /**
   * The module settings.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $settings;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * Constructs a TugboatExecute object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, FileSystemInterface $file_system) {
    $this->settings = $config_factory->get('tugboat.settings');
    $this->fileSystem = $file_system;
  }

  /**
   * Execute a Tugboat CLI command.
   *
   * @param $command_string
   *   The tugboat command string with any options. The --api-token and --json
   *   options are passed automatically. Tugboat command portion should be
   *   specified in this string, such as "find <id>" or "stop <id>".
   * @param array $return_data
   *   Returned data as an array if the command was successful.
   * @param $return_error_string
   *   A single error string if tugboat returned an error or if it is not
   *   possible to reach tugboat.
   * @param string $executable_path
   *   Optional. The path to the executable on the server. If not provided, the
   *   executable path provided in the config file will be used.
   * @param string $result_file_uri
   *   Optional.
   * @param string $output_file_uri
   *   Optional.
   * @param string $error_file_uri
   *   Optional.
   *
   * @return bool
   *   TRUE if the command executes without error, FALSE on error.
   */
  public function execute($command_string, array &$return_data, &$return_error_string, $executable_path = NULL, $result_file_uri = NULL, $output_file_uri = NULL, $error_file_uri = NULL) {
    if (empty($executable_path)) {
      $executable_path = $this->settings->get('executable_path');
    }

    if (!$this->isExecutable($executable_path, $return_error_string)) {
      return FALSE;
    }

    if (!$this->isSafeCommand($command_string, $return_error_string)) {
      return FALSE;
    }

    $command = $this->buildCommand($executable_path, $command_string, $result_file_uri);

    // Fire off the command via the binary file.
    $pipe_spec = [
      0 => ["pipe", "r"],  // stdin pipe to send input.
      1 => ["pipe", "w"],  // stdout pipe to receive output.
      2 => ["pipe", "w"]   // errors pipe to receive output.
    ];
    $pipes = [];

    // Check if the optional file path parameter is passed, and write to the
    // file while preserving the other streams.
    $process = proc_open($command, $pipe_spec, $pipes);
    fclose($pipes[0]);
    $std_output = stream_get_contents($pipes[1]);
    if (!empty($output_file_uri)) {
      $this->fileSystem->saveData($std_output, $output_file_uri);
    }
    $error_output = stream_get_contents($pipes[2]);
    fclose($pipes[1]);
    fclose($pipes[2]);
    $process_exit_code = proc_get_status($process)['exitcode'];
    proc_close($process);

    if ($error_output) {
      $return_error_string = trim($error_output);
      if (!empty($error_file_uri)) {
        $this->fileSystem->saveData($error_output, $error_file_uri);
      }
    }
    if ($std_output) {
      $decoded_json = json_decode($std_output, TRUE);
      if ($decoded_json === NULL) {
        // Work-around https://github.com/Lullabot/tugboat/issues/2999.
        // Use the last line of JSON output and ignore any progress information.
        if ($process_exit_code === 0) {
          $lines = explode("\n", $std_output);
          $last_line = end($lines);
          if ($decoded_json = json_decode($last_line, TRUE)) {
            $return_data = $decoded_json;
          }
        }

        $return_error_string = 'Unparseable JSON returned.';
      }
      else {
        $return_data = $decoded_json;
      }
    }

    return $process_exit_code === 0;
  }

  /**
   * Validate the executable path.
   *
   * @param string $executable_path
   *   The path to the executable on the server.
   * @param string $return_error_string
   *   An error string, passed by reference. It is only changed if there is an error.
   *
   * @return bool
   *   TRUE if the file exists and is executable.
   */
  public function isExecutable($executable_path, &$return_error_string) {
    if (!is_file($executable_path)) {
      $return_error_string = $this->t('No tugboat executable file found at the provided path.');
      return FALSE;
    }

    if (!is_executable($executable_path)) {
      $return_error_string = $this->t('The Tugboat CLI binary was found, but it is not executable.');
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Validate the tugboat command.
   *
   * Ensure input string is safe from any dangerous characters.
   * Characters allowed: 0-9, a-z, ., =, +, -, ', and a blank space.
   *
   * @param string $command_string
   *   The command to be passed to the Tugboat executable.
   * @param string $return_error_string
   *   An error string, passed by reference. It is only changed if there is an error.
   *
   * @return bool
   *   TRUE if the command is safe.
   */
  public function isSafeCommand($command_string, &$return_error_string) {
    if (!preg_match('/^[0-9a-z=+\-\' ]+$/', $command_string)) {
      $return_error_string = $this->t(
        'Invalid character for Tugboat command. String given: @string',
        ['@string' => $command_string]
      );
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Build the command to be executed.
   *
   * @param string $executable_path
   *   The path to the executable on the server.
   * @param string $command_string
   *   The command to be passed to the Tugboat executable.
   * @param string|null $result_file_uri
   *   A file where the output should be saved.
   *
   * @return string
   *   The full command to be executed.
   */
  protected function buildCommand($executable_path, $command_string, $result_file_uri) {
    $api_token = $this->settings->get('token');
    $command = "$executable_path --api-token='$api_token' $command_string --json";
    if (!empty($result_file_uri)) {
      $command .= ' 2>&1 | tee ' . $this->fileSystem->realpath($result_file_uri);
    }

    return $command;
  }

}
