<?php

namespace Drupal\turbo_sms_notify\Plugin\WebformHandler;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\turbo_sms_notify\Service\Sender;
use Drupal\webform\Plugin\WebformHandlerBase;
use Drupal\webform\Plugin\WebformHandlerInterface;
use Drupal\webform\WebformSubmissionInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Sends SMS/Viber via TurboSMS on webform submission save.
 *
 * @WebformHandler(
 *   id = "turbo_sms_notify",
 *   label = @Translation("TurboSMS notify"),
 *   category = @Translation("Notification"),
 *   description = @Translation("Sends SMS/Viber via TurboSMS upon submission."),
 *   cardinality = \Drupal\webform\Plugin\WebformHandlerInterface::CARDINALITY_UNLIMITED,
 *   results = TRUE
 * )
 */
class TurboSmsHandler extends WebformHandlerBase {

  /** @var \Drupal\turbo_sms_notify\Service\Sender */
  protected Sender $sender;

  /** @var \Drupal\Core\Config\ImmutableConfig */
  protected \Drupal\Core\Config\ImmutableConfig $cfg;

  /** @var \Psr\Log\LoggerInterface */
  protected LoggerInterface $logger;

  /**
   * DI factory.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   Service container.
   * @param array $configuration
   *   Plugin configuration.
   * @param string $plugin_id
   *   Plugin ID.
   * @param mixed $plugin_definition
   *   Plugin definition.
   *
   * @return static
   *   New instance.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    /** @var self $instance */
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->sender = $container->get('turbo_sms_notify.sender');
    $instance->cfg = $container->get('config.factory')->get('turbo_sms_notify.settings');
    $instance->logger = $container->get('logger.factory')->get('turbo_sms_notify');
    return $instance;
  }

  /**
   * Called after the submission is saved (works when results are stored).
   *
   * @param \Drupal\webform\WebformSubmissionInterface $submission
   *   Saved submission entity.
   * @param bool $update
   *   Whether this is an update operation (ignored).
   */
  public function postSave(WebformSubmissionInterface $submission, $update = TRUE) {
    // Debug entry for visibility.
    $this->logger->info('Handler: triggered for webform="@wf", sid=@sid', [
      '@wf' => $submission->getWebform()->id(),
      '@sid' => $submission->id(),
    ]);

    // Filter by configured webform (supports "*" to match any).
    $configured = (string) ($this->cfg->get('webform_id') ?? '');
    $current = $submission->getWebform()->id();
    if ($configured === '') {
      $this->logger->warning('Handler: webform_id is not configured; skipping.');
      return;
    }
    if ($configured !== '*' && $configured !== $current) {
      $this->logger->info('Handler: ignore "@curr" (configured "@cfg").', [
        '@curr' => $current, '@cfg' => $configured,
      ]);
      return;
    }

    // Resolve phone from submission data.
    $data = $submission->getData();
    $phone_key = (string) ($this->cfg->get('phone_field') ?? 'phone');
    $raw = $data[$phone_key] ?? NULL;
    if (is_array($raw)) {
      // Support common composite structures like ['value' => '...'].
      $raw = $raw['value'] ?? reset($raw) ?? NULL;
    }
    $phone = $this->normalizePhone((string) $raw);
    if (!$phone) {
      $this->logger->warning('Handler: invalid phone "@field" for @sid. Keys: @keys', [
        '@field' => $phone_key,
        '@sid'   => $submission->id(),
        '@keys'  => implode(', ', array_keys($data)),
      ]);
      return;
    }

    // Build message text from template + tokens (if token service available).
    $text = $this->buildText($submission, $phone);

    // Delegate actual sending to the service (reads channel/alpha/etc. from config).
    $ok = $this->sender->send($phone, $text);

    // Optional: add context log line (Sender already logs result).
    if (!$ok) {
      $this->logger->error('Handler: send() returned FALSE for @p (webform="@wf")', [
        '@p' => $phone, '@wf' => $current,
      ]);
    }
  }

  /**
   * Build a message text from config template and submission data.
   *
   * @param \Drupal\webform\WebformSubmissionInterface $submission
   *   Submission entity.
   * @param string $phone
   *   Normalized recipient phone.
   *
   * @return string
   *   Message text.
   */
  protected function buildText(WebformSubmissionInterface $submission, string $phone): string {
    $data = $submission->getData();
    $template = (string) ($this->cfg->get('message_template') ?? '');

    // First, simple placeholders for backward compatibility.
    $text = strtr($template, [
      '[name]'  => (string) ($data['name'] ?? ($data['name_customer'] ?? '')),
      '[phone]' => $phone,
    ]);

    // Token API replacement (if token module is enabled).
    // Supports tokens like [site:name], [current-user:mail], [date:custom:Y-m-d],
    // [webform_submission:values:phone], [webform:title], etc.
    if (\Drupal::hasService('token')) {
      try {
        $contexts = [
          'webform_submission' => $submission,
          'webform'            => $submission->getWebform(),
          // 'site' и 'current-user' токены обычно не требуют явного контекста.
        ];
        $text = \Drupal::token()->replace($text, $contexts, [
          'clear' => TRUE,   // remove unknown tokens if any
        ]);
      }
      catch (\Throwable $e) {
        // Ignore token errors; keep plain text.
        $this->logger->warning('Handler: token replacement failed: @m', ['@m' => $e->getMessage()]);
      }
    }

    return $text !== '' ? $text : 'Thank you! We will contact you shortly.';
  }

  /**
   * Normalize phone to E.164-ish with leading '+'.
   *
   * @param string $phone
   *   Raw phone input.
   *
   * @return string|null
   *   Normalized phone or NULL if invalid.
   */
  protected function normalizePhone(string $phone): ?string {
    if ($phone === '') {
      return NULL;
    }
    $phone  = preg_replace('/[^+\d]/', '', $phone) ?? '';
    $digits = preg_replace('/\D/', '', $phone) ?? '';
    if (strlen($digits) < 10) {
      return NULL;
    }
    if ($phone[0] !== '+') {
      $phone = '+' . $digits;
    }
    return $phone;
  }

}
